module Viterbi

export run, backtrack, annotate

function viterbiloop(dna::String, N::Int64, K::Int64, symbolIndex::Dict,
		A::Array{Float64}, phi::Array{Float64}, w::Array{Float64})
	for n = 2:N
		x = symbolIndex[string(dna[n])] + 1.

		for k = 1:K
			m = -Inf

			if phi[k,x] == 0
				continue
			end

			for j = 1:K
				if w[j,n-1] == -Inf || A[j,k] == 0
					continue
				end

				v = w[j,n-1] + log(A[j,k])
				if v > m
					m = v
				end
			end

			w[k,n] = log(phi[k,x]) + m
		end

	end
end

function run(hmm, dna::String)
	N = length(dna)
	K = length(hmm["states"])
	A = hmm["A"]
	symbolIndex = hmm["aIndex"]

	phi = hmm["phi"]

	w = ones(Float64, (K, N)) * -Inf

	for k = 1:K
        x = hmm["aIndex"][string(dna[1])] + 1.
        w[k,1] = log(phi[k,x])
	end

	viterbiloop(dna, N, K, symbolIndex, A, phi, w)

	return w
end

function backtrackloop(dna::String, N::Int64, K::Int64, symbolIndex::Dict,
	A::Array{Float64}, phi::Array{Float64}, w::Array{Float64}, z::Array{Int64})
	for n = N-1:-1:1
        x = symbolIndex[string(dna[n+1])] + 1

        # Look for the state which sums to the maximum value for n+1
		for k = 1:K
            if w[k, n] == -Inf || A[k, z[n+1]] == 0 || phi[z[n+1], x] == 0
                continue
			end

            if w[k, n] + log(A[k, z[n+1]]) + log(phi[z[n+1], x]) == w[z[n+1], n+1]
                z[n] = k
                break
			end
		end
	end
end

function backtrack(hmm, dna, w)
    N  =  length(dna)
    K  =  length(hmm["states"])
    A  =  hmm["A"]
    phi = hmm["phi"]
	symbolIndex = hmm["aIndex"]

	z = zeros(Int64, N)
	(v, i) = findmax(w[:,end])

	z[end] = i

	backtrackloop(dna, N, K, symbolIndex, A, phi, w, z)

    return z
end

function annotate(hmm, z, outputfile)
	trans = Dict()
	for (k, v) in hmm["states"]
		trans[v["i"]+1] = v["a"]
	end

	stream = open(outputfile, "w+")
    write(stream, ">Annotation generated by dat7\n")

	i = 0
	for n = 1:length(z)
		write(stream, trans[z[n]])

		i += 1
		if i == 60
			write(stream, "\n")
			i = 0
		end
	end

	close(stream)
end

end
