#define _DEFAULT_SOURCE
#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <inttypes.h>
#include <streams.h>
#include <string.h>
#include <time.h>
#include <getopt.h>
#include <sys/resource.h>

#include "dmerge.h"
#include "mergesort.h"

#define D_DEFAULT 8
#define N_DEFAULT 32768

int STREAM_TYPE = 3;
int BUFFER = 1024;

void printusage(char *argv[]) {
	fprintf(stderr, "Usage: %s \n"
			"\t[-N #samples]\n"
			"\t[-M memory size]\n"
			"\t[-B buffer size]\n"
			"\t[-d merge d streams at a time]\n"
			"\t[-S stream implementation (1-4 = A-D)]\n"
			, argv[0]);
}

int main(int argc, char *argv[]) {
	char *inputfile = "BIGFILE";
	char *outputfile = "sorted.out";
	struct timespec begin;
	struct timespec end;
	int opt;
	int total_ns;
	int total_s;
	struct rusage usage;

	unsigned int d = 9;
	uint64_t N = 0;
	uint64_t M = 100*1024*1024;

	while ((opt = getopt(argc, argv, "N:M:d:B:S:i:")) != -1) {
		switch (opt) {
		case 'N':
			N = strtoull(optarg, NULL, 10);
			break;
		case 'M':
			M = strtoull(optarg, NULL, 10);
			break;
		case 'B':
			BUFFER = strtol(optarg, NULL, 10);
			break;
		case 'S':
			STREAM_TYPE = strtol(optarg, NULL, 10);
			if(STREAM_TYPE > 4 || STREAM_TYPE < 1) {
				printusage(argv);
				exit(EXIT_FAILURE);
			}
			break;
		case 'd':
			d = strtoul(optarg, NULL, 10);
			break;
		case 'i':
			inputfile = strndup(optarg, 200);
			break;
		default:
			printusage(argv);
			exit(EXIT_FAILURE);
		}
	}

	if (N == 0) {
		N = filesize(inputfile);
	}

	clock_gettime(CLOCK_REALTIME, &begin);

	uint64_t fhs = N/M; // The amount of filehandles we need, -4.
	if(fhs > 1019) {
		fprintf(stderr, "M is too small for N, we would open %" PRIu64 " filehandles\n", fhs);
		fprintf(stderr, "Min M is %" PRIu64 "\n", N/1019);
		exit(EXIT_FAILURE);
	}

	in_stream in = (in_stream) {
		.next   = bi_next,
		.eos    = bi_eos,
		.close  = bi_close,
		.open   = open_input_stream
	};

	out_stream out = (out_stream) {
		.write  = bo_write,
		.close  = bo_close,
		.create = open_output_stream
	};

	mergesort(inputfile, outputfile, &in, &out, N, M, d);

	getrusage(RUSAGE_THREAD, &usage);
	clock_gettime(CLOCK_REALTIME, &end);

	clock_gettime(CLOCK_REALTIME, &end);

	if (begin.tv_nsec <= end.tv_nsec) {
		total_ns = end.tv_nsec - begin.tv_nsec;
		total_s  = end.tv_sec  - begin.tv_sec;
	} else {
		total_ns = end.tv_nsec + (1e9 - begin.tv_nsec);
		total_s  = end.tv_sec - begin.tv_sec - 1;
	}

	printf(
		"%d,%d,%ld,%ld,%ld,%ld,%ld,%ld,%ld,%ld,%ld,%ld,%ld,%ld,%ld,%ld\n",
			/* Total realworld seconds/nsec used */
			total_s,
			total_ns,
			/* Total amount of CPU user time used.  */
			usage.ru_utime.tv_sec,   // Seconds
			usage.ru_utime.tv_usec,  // Microseconds
			/* Total amount of CPU system time used.  */
			usage.ru_stime.tv_sec,   // Seconds
			usage.ru_stime.tv_usec,  // Microseconds
			/* Maximum resident set size (in kilobytes).  */
			usage.ru_maxrss,
			/* Amount of data segment memory used (kilobyte-seconds).  */
			usage.ru_idrss,
			/* Amount of stack memory used (kilobyte-seconds).  */
			usage.ru_isrss,
			/* Number of soft page faults (i.e. those serviced by reclaiming
			   a page from the list of pages awaiting reallocation.  */
			usage.ru_minflt,
			/* Number of hard page faults (i.e. those that required I/O).  */
			usage.ru_majflt,
			/* Number of times a process was swapped out of physical memory. */
			usage.ru_nswap,
			/* Number of input operations via the file system.  Note: This
			   and `ru_oublock' do not include operations with the cache.  */
			usage.ru_inblock,
			/* Number of output operations via the file system.  */
			usage.ru_oublock,
			/* Number of voluntary context switches, i.e. because the process
			   gave up the process before it had to (usually to wait for some
			   resource to be available).  */
			usage.ru_nvcsw,
			/* Number of involuntary context switches, i.e. a higher priority
			 * process became runnable or the current process used up its time
			 * slice. */
			usage.ru_nivcsw
		);

	return EXIT_SUCCESS;
}
