#define _DEFAULT_SOURCE

#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <inttypes.h>
#include <math.h>
#include <measure.h>

#include "xutil.h"
#include "bitvector.h"
#include "naive.h"
#include "naive_popcount.h"
#include "classic.h"
#include "popcount.h"
#include "clark.h"

int main(int argc, char *argv[]) {
	(void) argc;
	(void) argv;

	uint64_t i, j, k, l, m;
	uint64_t size;
	uint64_t FRACTION = 8;
	uint64_t INNER_ITERATIONS = 20;
	uint64_t MIDDLE_ITERATIONS = 5;
	uint64_t OUTER_ITERATIONS = 10;
	char buf[100];
	struct succinct_t *succ[MIDDLE_ITERATIONS][INNER_ITERATIONS];
 
	if (!measure_init()) {
		xerror("Failed initializing measure", __LINE__, __FILE__);
	}

	for (m = 0; m < MIDDLE_ITERATIONS; m++) {
		for (k = 0; k < INNER_ITERATIONS; k++) {
			succ[m][k] = xmalloc(sizeof(struct succinct_t));
		}
	}

	srand48(42);

	for (l = 0; l < OUTER_ITERATIONS; l++) {
		for (i = 3; i < 32; i++) {
			for (j = 0; j < FRACTION; j++) {
				size = floor(pow(2, i)*(1+((double)j/FRACTION)));
				sprintf(buf, "%"PRIu64, size);

				bitvector_t *B = bitvector_create(size);
				bitvector_randomize(B);

				for (m = 0; m < MIDDLE_ITERATIONS; m++) {
					for (k = 0; k < INNER_ITERATIONS; k++) {
						succ[m][k]->B = B;
						succ[m][k]->i = ((uint64_t)mrand48())%size >> 1;
					}
				}

				// NÄIVE
				for (m = 0; m < MIDDLE_ITERATIONS; m++) {
					measure("Naive Rank", buf, (testfunc)succinct_naive_rank, (void *)&(succ[m]), INNER_ITERATIONS);
				}

				for (m = 0; m < MIDDLE_ITERATIONS; m++) {
					measure("Naive Select", buf, (testfunc)succinct_naive_select, (void *)&(succ[m]), INNER_ITERATIONS);
				}

				// NÄIVE POPCOUNT
				for (m = 0; m < MIDDLE_ITERATIONS; m++) {
					measure("Naive Popcount Rank", buf, (testfunc)succinct_naive_popcount_rank, (void *)&(succ[m]), INNER_ITERATIONS);
				}

				for (m = 0; m < MIDDLE_ITERATIONS; m++) {
					measure("Naive Popcount Select", buf, (testfunc)succinct_naive_popcount_select, (void *)&(succ[m]), INNER_ITERATIONS);
				}

				// CLASSIC
				succinct_classic_preprocess(B);

				for (m = 0; m < MIDDLE_ITERATIONS; m++) {
					measure("Classic Rank", buf, (testfunc)succinct_classic_rank, (void *)&(succ[m]), INNER_ITERATIONS);
				}

				for (m = 0; m < MIDDLE_ITERATIONS; m++) {
					measure("Classic Select", buf, (testfunc)succinct_classic_select, (void *)&(succ[m]), INNER_ITERATIONS);
				}

				succinct_classic_postprocess(B);

				// POPCOUNT
				succinct_popcount_preprocess(B);

				for (m = 0; m < MIDDLE_ITERATIONS; m++) {
					measure("Popcount Rank", buf, (testfunc)succinct_popcount_rank, (void *)&(succ[m]), INNER_ITERATIONS);
				}

				for (m = 0; m < MIDDLE_ITERATIONS; m++) {
					measure("Popcount Select", buf, (testfunc)succinct_popcount_select, (void *)&(succ[m]), INNER_ITERATIONS);
				}

				succinct_popcount_postprocess(B);

				// CLARK
				succinct_clark_preprocess(B);

				for (m = 0; m < MIDDLE_ITERATIONS; m++) {
					measure("Clark Select", buf, (testfunc)succinct_clark_select, (void *)&(succ[m]), INNER_ITERATIONS);
				}

				succinct_clark_postprocess(B);

				bitvector_destroy(B);
			}
		}
	}
	
	for (k = 0; k < MIDDLE_ITERATIONS; k++) {
		for (k = 0; k < INNER_ITERATIONS; k++) {
			if (NULL != (succ[m])[k]) {
				free((succ[m])[k]);
			}
		}
	}

	measure_destroy();

	return EXIT_SUCCESS;
}
