#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <inttypes.h>
#include <math.h>
#include <assert.h>

#include "bitvector.h"
#include "xutil.h"

bitvector_t *bitvector_create(uint64_t bits) {
	assert(bits > 0);

	uint64_t N = ceil((double)bits/WORD);
	uint64_t size = N*sizeof(bitvector);

	bitvector_t *B = malloc(sizeof(bitvector_t));

	B->B = xmalloc(size);
	B->N = N;
	B->size = size;
	B->bits = bits;

	memset(B->B, '\0', size);

	return B;
}

void bitvector_destroy(bitvector_t *bv) {
	if (NULL != bv) {
		if (NULL != bv->B) {
			free(bv->B);
		}
		free(bv);
	}
}

void bitvector_randomize(bitvector_t *bv) {
	uint64_t i;

	for (i = 0; i < bv->N; i++) {
		bv->B[i] = ((uint64_t)rand() << 32) | rand();
	}
}

void bitvector_randomize_zero(bitvector_t *bv, uint64_t factor) {
	uint64_t i;

	for (i = 0; i < bv->N; i++) {
		if ( (RAND_MAX/factor) > (uint32_t)rand() ) {
			bv->B[i] = ((uint64_t)rand() << 32) | rand();
		} else {
			bv->B[i] = 0;
		}
	}
}

void bitvector_print(bitvector_t *bv) {
	uint64_t i, k, c = 1;
	int j;
	unsigned char bytes[8];
	unsigned char byte;

	for (i = 0; i < bv->N; i++) {
		bytes[0] = (bv->B[i] >> 56) & 0xff;
		bytes[1] = (bv->B[i] >> 48) & 0xff;
		bytes[2] = (bv->B[i] >> 40) & 0xff;
		bytes[3] = (bv->B[i] >> 32) & 0xff;
		bytes[4] = (bv->B[i] >> 24) & 0xff;
		bytes[5] = (bv->B[i] >> 16) & 0xff;
		bytes[6] = (bv->B[i] >>  8) & 0xff;
		bytes[7] = (bv->B[i] >>  0) & 0xff;

		for (k = 0; k < sizeof(bitvector); k++) {
			for (j = 7; j >= 0 && c <= bv->bits; j--) {
				byte = bytes[k];
				byte &= (1 << j);
				byte >>= j;
				printf("%u", byte);
				c++;
			}
			printf(" ");
		}
		printf("\n");
	}

	printf("\n\n");
}

void bitvector_set_bits(bitvector *restrict bv, uint64_t start, 
		uint64_t width, bitvector value) { 
	uint64_t start_word = start / WORD;
	uint64_t end_word = (start+width-1) / WORD;
	uint64_t start_bit = start % WORD;
	uint64_t temp;

	// If we are assigning bits to one word only
	if ( start_word == end_word ) {
		if (width == WORD) {
			bv[start_word] = value;
		} else {
			temp = (WORD-(start_bit+width));
			bv[start_word] &= ~((((bitvector)1 << width)-1) << temp);
			bv[start_word] |= ((bitvector)value << temp);
		}
		return;
	}

	// If we are assigning over two consecutive words
	temp = width-(WORD-start_bit);

	bv[start_word] &= ~(((bitvector)1 << (WORD-start_bit))-1);
	bv[start_word] |= value >> temp;

	bv[end_word]   &= ((bitvector)1 << (WORD-temp))-1;
	bv[end_word]   |= ((bitvector)value << (WORD-temp));
}

bitvector bitvector_get_bits(bitvector *restrict bv, uint64_t start, 
		uint64_t width) { 
	uint64_t start_word = start / WORD;
	uint64_t end_word = (start+width-1) / WORD;
	uint64_t start_bit = start % WORD;
	uint64_t temp;

	bitvector bt[2] = {
		bv[start_word], 
		bv[end_word]
	};

	if (start_word == end_word) {
		if (width == WORD) {
			return bt[0];
		}
		temp = (WORD-(start_bit+width));
		bt[0] &= ((((bitvector)1 << width)-1) << temp);
		return bt[0] >> temp;
	}

	// If we are getting from two consecutive words
	temp = width-(WORD-start_bit);

	bt[0] &= ((bitvector)1 << (WORD-start_bit))-1;
	bt[1] >>= WORD-temp;

	return ((bitvector)bt[0] << temp) | bt[1];
}
