#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <inttypes.h>
#include <math.h>
#include <assert.h>
#include <getopt.h>

#include "../xutil.h"
#include "../bitvector.h"
#include "../naive.h"
#include "../naive_popcount.h"
#include "../classic.h"
#include "../popcount.h"
#include "../clark.h"

void printusage(char *argv[]) {
	fprintf(stderr, "Usage: %s \n"
			"\t[-b bit combinations up till b (6 .. b)]\n"
			"\t[-s specifict amount of bits]\n"
			, argv[0]);
}

int main(int argc, char *argv[]) {
	int opt;
	uint64_t i, k;
	uint64_t sizes = 1024;
	uint64_t spc = 0;
	uint64_t j = 1;
	uint64_t rank;
	uint64_t select;
	struct succinct_t succ;

	while ((opt = getopt(argc, argv, "b:j:s:")) != -1) {
		switch (opt) {
		case 's':
			spc = strtoull(optarg, NULL, 10);
			break;
		case 'b':
			sizes = strtoull(optarg, NULL, 10);
			break;
		default:
			printusage(argv);
			exit(EXIT_FAILURE);
		}
	}

	if ( !spc ) {
		bitvector_t *B = bitvector_create(sizes);
		bitvector_randomize(B);

		succ.B = B;

		for (i = 6; i <= sizes; i++) {
			B->bits = i;
			B->N = ceil((double)i/WORD);
			B->size = B->N*sizeof(bitvector);

			if ( i % 1000 == 0 ) {
				printf("Bits: %"PRIu64"\n", i);
			}

			for (j = 0; j <= i; j++) {
				succ.i = j;
				assert(succinct_naive_rank(&succ) == succinct_naive_popcount_rank(&succ));
				assert(succinct_naive_select(&succ) == succinct_naive_popcount_select(&succ));
			}

			succinct_classic_preprocess(B);
			for (j = 0; j <= i; j++) {
				succ.i = j;
				assert(succinct_naive_rank(&succ) == succinct_classic_rank(&succ));
				assert(succinct_naive_select(&succ) == succinct_classic_select(&succ));
			}
			succinct_classic_postprocess(B);
			succinct_popcount_preprocess(B);
			for (j = 0; j <= i; j++) {
				succ.i = j;
				assert(succinct_naive_rank(&succ) == succinct_popcount_rank(&succ));
				assert(succinct_naive_select(&succ) == succinct_popcount_select(&succ));
			}
			succinct_popcount_postprocess(B);
			succinct_clark_preprocess(B);
			for (j = 0; j <= i; j++) {
				succ.i = j;
				assert(succinct_naive_select(&succ) == succinct_clark_select(&succ));
			}
			succinct_clark_postprocess(B);
		}

		bitvector_destroy(B);
	} else {
		k = 17%spc;
		
		bitvector_t *B = bitvector_create(spc);
		bitvector_randomize(B);

		succ.B = B;
		succ.i = k;

		printf("================ BITVECTOR ================\n\n");
		bitvector_print(B);
		printf("===========================================\n\n");

		printf("================   Naive   ================\n\n");

		rank = 0;
		select = 0;
		rank = succinct_naive_rank(&succ);
		select = succinct_naive_select(&succ);

		printf("Counted %"PRIu64" occurences of 1 up till offset %"PRIu64"\n", 
				rank, k);

		printf("Offset of the %"PRIu64"'th occurence of 1 is %"PRIu64"\n", 
				k, select);

		printf("===========================================\n\n");


		printf("============  Naive Popcount   ============\n\n");

		rank = 0;
		select = 0;

		rank = succinct_naive_popcount_rank(&succ);
		select = succinct_naive_popcount_select(&succ);

		printf("Counted %"PRIu64" occurences of 1 up till offset %"PRIu64"\n", 
				rank, k);

		printf("Offset of the %"PRIu64"'th occurence of 1 is %"PRIu64"\n", 
				k, select);

		for (j = 0; j <= spc; j++) {
			succ.i = j;
			if ( j % 10000 == 0 ) {
				printf("\tj: %"PRIu64"\n", j);
			}
			assert(succinct_naive_rank(&succ) == succinct_naive_popcount_rank(&succ));
			assert(succinct_naive_select(&succ) == succinct_naive_popcount_select(&succ));
		}

		printf("===========================================\n\n");

		printf("================  Classic  ================\n\n");
		succinct_classic_preprocess(B);
		
		succ.i = k;
		rank = 0;
		select = 0;
		rank = succinct_classic_rank(&succ);
		select = succinct_classic_select(&succ);
		
		printf("Counted %"PRIu64" occurences of 1 up till offset %"PRIu64"\n", 
				rank, k);

		printf("Offset of the %"PRIu64"'th occurence of 1 is %"PRIu64"\n", 
				k, select);

		for (j = 0; j <= spc; j++) {
			succ.i = j;
			if ( j % 10000 == 0 ) {
				printf("\tj: %"PRIu64"\n", j);
			}
			assert(succinct_naive_rank(&succ) == succinct_classic_rank(&succ));
			assert(succinct_naive_select(&succ) == succinct_classic_select(&succ));
		}

		succinct_classic_postprocess(B);
		printf("===========================================\n\n");

		printf("================  Popcount  ================\n\n");
		succinct_popcount_preprocess(B);

		succ.i = k;
		rank = 0;
		select = 0;
		rank = succinct_popcount_rank(&succ);
		select = succinct_popcount_select(&succ);
		
		printf("Counted %"PRIu64" occurences of 1 up till offset %"PRIu64"\n", 
				rank, k);

		printf("Offset of the %"PRIu64"'th occurence of 1 is %"PRIu64"\n", 
				k, select);

		for (j = 0; j <= spc; j++) {
			succ.i = j;
			if ( j % 10000 == 0 ) {
				printf("\tj: %"PRIu64"\n", j);
			}
			assert(succinct_naive_rank(&succ) == succinct_popcount_rank(&succ));
			assert(succinct_naive_select(&succ) == succinct_popcount_select(&succ));
		}

		succinct_popcount_postprocess(B);
		printf("===========================================\n\n");

		printf("=================  Clark  =================\n\n");
		succinct_clark_preprocess(B);

		succ.i = k;
		select = 0;
		select = succinct_clark_select(&succ);

		printf("Offset of the %"PRIu64"'th occurence of 1 is %"PRIu64"\n", 
				k, select);

		for (j = 0; j <= spc; j++) {
			succ.i = j;
			if ( j % 10000 == 0 ) {
				printf("\tj: %"PRIu64"\n", j);
			}
			assert(succinct_naive_select(&succ) == succinct_clark_select(&succ));
		}

		succinct_clark_postprocess(B);
		printf("===========================================\n\n");

		bitvector_destroy(B);
	}

	printf("SUCCESS\n");

	return EXIT_SUCCESS;
}
