#include <inttypes.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <measure.h>

#include "xutil.h"
#include "matrix.h"
#include "drepper.h"
#include "naive.h"
#include "naive_idx.h"
#include "frigo.h"
#include "strassen.h"
#include "strassen_idx.h"

void printusage(char *argv[]) {
	fprintf(stderr, "Usage: %s \n"
			"\t[-n (Run Naive)]\n"
			"\t[-s (Run Strassen)]\n"
			"\t[-f (Run Frigo)]\n"
			"\t[-p (Run NaiveIDX)]\n"
			"\t[-d (Run Drepper)]\n"
			"\t[-q (Run StrassenIDX)]\n"
			"\t[-b Start with i = <PARAMATER>]\n"
			"\t[-i End after this many iterations]\n"
			, argv[0]);
}

int main(int argc, char *argv[]) {
	int i, j, k, opt,
		REPS = 10,
		REPS_MEASURE = 1,
		TESTRUNS = 8,
		BASE = 6;
	uint32_t size;
	char buf[100];
	unsigned int n=0,s=0,f=0,q=0,p=0,d=0,o=0;

	while ((opt = getopt(argc, argv, "nsfpqdi:b:or:")) != -1) {
		switch (opt) {
		case 'n':
			n = 1;
			break;
		case 's':
			s = 1;
			break;
		case 'd':
			d = 1;
			break;
		case 'f':
			f = 1;
			break;
		case 'o':
			o = 1;
			break;
		case 'p':
			p = 1;
			break;
		case 'q':
			q = 1;
			break;
		case 'i':
			TESTRUNS = atoi(optarg);
			break;
		case 'b':
			BASE = atoi(optarg);
			break;
		case 'r':
			REPS = atoi(optarg);
			break;
		default:
			printusage(argv);
			exit(EXIT_FAILURE);
		}
	}

	if (!measure_init()) {
		xerror("Failed initializing measure", __LINE__, __FILE__);
	}

	if (!n && !s && !f && !p && !q && !d) {
		n=1;s=1;f=1;q=1;p=1;d=1;
	}

	matrix_t *A;
	matrix_t *B;
	matrix_t *C;

	struct matrix_mult *str[REPS_MEASURE];
	struct strassen_mult *str_str[REPS_MEASURE];

	for (i = 0; i < REPS_MEASURE; i++) {
		str[i] = malloc(sizeof(struct matrix_mult));
		str_str[i] = malloc(sizeof(struct strassen_mult));
	}

	uint32_t fractions = 8;

	for (i = 0; i < TESTRUNS; i ++) {
		uint32_t N = pow(2, BASE+i);

		for (k = 1; k <= (int)fractions; k++) {
			if (o == 0) {
				// Only use fractions if o flag is not set.
				size = N/2 * (1.0 + (double)k/fractions);
			} else {
				size = N;
			}

			A = matrix_create(size, size);
			B = matrix_create(size, size);
			C = matrix_create(size, size);

			matrix_randomize(A, (unsigned int)rand());
			matrix_randomize(B, (unsigned int)rand());

			for (j = 0; j < REPS_MEASURE; j++) {
				str[j]->A = A;
				str[j]->B = B;
				str[j]->C = C;

				str_str[j]->A = A;
				str_str[j]->B = B;
				str_str[j]->C = C;
				str_str[j]->th = 0;
			}

			for (j = 0; j < REPS; j++) {
				sprintf(buf, "%d,%d", size, size);

				if (n) {
					matrix_zero(C);
					measure("Naive",       buf, (testfunc)naive,        (void*)&str,     REPS_MEASURE);
				}

				if (s) {
					matrix_zero(C);
					measure("Strassen",    buf, (testfunc)strassen,     (void*)&str_str, REPS_MEASURE);
				}

				if (f) {
					matrix_zero(C);
					measure("Frigo",       buf, (testfunc)frigo,        (void*)&str,     REPS_MEASURE);
				}

				if (p) {
					matrix_zero(C);
					measure("NaiveIDX",    buf, (testfunc)naive_idx,    (void*)&str,     REPS_MEASURE);
				}

				if (q) {
					matrix_zero(C);
					measure("StrassenIDX", buf, (testfunc)strassen_idx, (void*)&str_str, REPS_MEASURE);
				}

				if (d) {
					matrix_zero(C);
					measure("Drepper",     buf, (testfunc)drepper,      (void*)&str,     REPS_MEASURE);
				}
			}

			matrix_destroy(A);
			matrix_destroy(B);
			matrix_destroy(C);

			if (o == 1) {
				break;
			}
		}
	}

	for (i = 0; i < REPS_MEASURE; i++) {
		free(str[i]);
		free(str_str[i]);
	}

	return EXIT_SUCCESS;
}


