#define _DEFAULT_SOURCE
#define _GNU_SOURCE
#define _BSD_SOURCE

#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <inttypes.h>
#include <string.h>
#include <getopt.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <dirent.h>

#include "../xutil.h"
#include "../inorder.h"
#include "../bfs.h"
#include "../dfs.h"
#include "../veb.h"

void printusage(char *argv[]) {
	fprintf(stderr, "Usage: %s \n"
			"\t[-i iterations]\n"
			, argv[0]);
}

int main(int argc, char *argv[]) {
	int opt;
	int fd;

	uint32_t iterations = 0;
	uint32_t size = 0;
	uint32_t N = 0;
	uint32_t res[4];
	uint32_t i, j, x;

	uint32_t *storage;
	inorder_t *inorder;
	bfs_t *bfs;
	dfs_t *dfs;
	veb_t *veb;

	char **files;
	uint32_t files_count;
	DIR *dir;
	struct dirent *entry;

	while ((opt = getopt(argc, argv, "i:")) != -1) {
		switch (opt) {
		case 'i':
			iterations = strtoull(optarg, NULL, 10);
			break;
		default:
			printusage(argv);
			exit(EXIT_FAILURE);
		}
	}

	if (!iterations) {
		printusage(argv);
		exit(EXIT_FAILURE);
	}

	files_count = xfilesdir("../DATA");
	files = xcalloc(files_count, sizeof(char *));

	if ((dir = opendir("../DATA")) == NULL) {
		xerror(strerror(errno), __LINE__, __FILE__);
	}

	i = 0;
	while ( (entry = readdir(dir)) != NULL ) {
		if (entry->d_type == DT_REG) { /* If the entry is a regular file */
			files[i] = xmalloc(sizeof(char)*(256+8));
			memcpy(files[i], "../DATA/", 8);
			memcpy(&files[i][8], entry->d_name, 256);
			i++;	
		}
	}

	closedir(dir);

	for (i = 0; i < files_count; i++) {
		size = xfilesize(files[i]);
		N = size/sizeof(uint32_t);

		// Allocate memory to hold integers
		storage = xmalloc(size);

		// Open file containing integers and store file descriptor
		fd = open(files[i], O_RDONLY);
		free(files[i]);

		// Error check
		if (fd == -1) {
			xerror(strerror(errno), __LINE__, __FILE__);
		}

		// Read entire file and error check 
		if ( read(fd, storage, size) == -1 ) {
			xerror(strerror(errno), __LINE__, __FILE__);
		}

		// Close file descriptor
		close(fd);

		// Creating data structures
		inorder = inorder_create(storage, N, 0.5);
		bfs = bfs_create(inorder->storage, N, 0.5);
		dfs = dfs_create(inorder->storage, N, 0.5);
		veb = veb_create(inorder->storage, N);

		// No need for this anymore
		free(storage);

		struct inorder_search inorder_str;
		struct bfs_search bfs_str;
		struct dfs_search dfs_str;
		struct veb_search veb_str;

		inorder_str.inorder = inorder;
		bfs_str.bfs = bfs;
		dfs_str.dfs = dfs;
		veb_str.veb = veb;

		for (j = 0; j < iterations; j++) {
			x = (uint32_t)mrand48();

			inorder_str.x = x;
			bfs_str.x = x;
			dfs_str.x = x;
			veb_str.x = x;

			res[0] = inorder_pred(&inorder_str);
			res[1] = bfs_pred(&bfs_str);
			res[2] = dfs_pred(&dfs_str);
			res[3] = veb_pred(&veb_str);

			// If they are not equal
			if (res[0] != res[1] || res[0] != res[2] || res[0] != res[3]) {
				// Check that we do not search for element that is not present
				// and is smaller than the smallest element in the tree
				if (x > res[0] || x > res[1] || x > res[2] || x > res[3]) {
					xerror(
						"The binary search of the different structures did "
						"not yield the same result", 
						__LINE__, 
						__FILE__
					);
				}
			}
		}

		free(inorder->storage);
		free(inorder);
		free(bfs->storage);
		free(bfs);
		free(dfs->storage);
		free(dfs);
		free(veb->table);
		free(veb->storage);
		free(veb);
	}

	free(files);

	printf("SUCCESS\n");

	// Exit program successfully
	return EXIT_SUCCESS;
}
