#include <string.h>
#include <stdio.h>
#include <inttypes.h>
#include <criterion/criterion.h>

#include "draw_binary.h"
#include "binaryheap.h"
#include "node.h"

BinaryHeap_t *heap;

void setup(void) {
	heap = MakeBinaryHeap();
}

void teardown(void) {
	FreeBinaryHeap(heap);	
}

TestSuite(binary_heap, .init=setup, .fini=teardown);

Test(binary_heap, empty_heap) {
	cr_expect(NULL == BinaryHeapDeleteMin(heap), "Testing DeleteMin returns NULL if heap is empty");
	cr_expect(NULL == BinaryHeapFindMin(heap), "Testing FindMin returns NULL if heap is empty");
}

Test(binary_heap, insert_and_findmin) {
	Node_t **A;
	Node_t *val;

	BinaryHeapInsert(heap, -328);

	val = BinaryHeapFindMin(heap);
	cr_expect(-328   == val->key, "Expected minimum to be -328 got %"PRId64, val->key);

	BinaryHeapInsert(heap, 394);

	val = BinaryHeapFindMin(heap);
	cr_expect(-328   == val->key, "Expected minimum to be -328 got %"PRId64, val->key);

	BinaryHeapInsert(heap, 294);

	val = BinaryHeapFindMin(heap);
	cr_expect(-328   == val->key, "Expected minimum to be -328 got %"PRId64, val->key);

	BinaryHeapInsert(heap, -32489);

	val = BinaryHeapFindMin(heap);
	cr_expect(-32489   == val->key, "Expected minimum to be -328 got %"PRId64, val->key);

	BinaryHeapInsert(heap, 38);

	val = BinaryHeapFindMin(heap);
	cr_expect(-32489   == val->key, "Expected minimum to be -328 got %"PRId64, val->key);

	BinaryHeapInsert(heap, 0);

	val = BinaryHeapFindMin(heap);
	cr_expect(-32489   == val->key, "Expected minimum to be -328 got %"PRId64, val->key);

	BinaryHeapInsert(heap, 4000);

	val = BinaryHeapFindMin(heap);
	cr_expect(-32489   == val->key, "Expected minimum to be -328 got %"PRId64, val->key);

	BinaryHeapInsert(heap, 4000);

	val = BinaryHeapFindMin(heap);
	cr_expect(-32489   == val->key, "Expected minimum to be -328 got %"PRId64, val->key);

	A = heap->A;
		
	cr_expect(-32489 == A[0]->key, "Expected -32489 got %"PRId64, A[0]->key);
	cr_expect(-328   == A[1]->key, "Expected -328 got %"PRId64,   A[1]->key);
	cr_expect(0      == A[2]->key, "Expected 0 got %"PRId64,      A[2]->key);
	cr_expect(394    == A[3]->key, "Expected 394 got %"PRId64,    A[3]->key);
	cr_expect(38     == A[4]->key, "Expected 38 got %"PRId64,     A[4]->key);
	cr_expect(294    == A[5]->key, "Expected 294 got %"PRId64,    A[5]->key);
	cr_expect(4000   == A[6]->key, "Expected 4000 got %"PRId64,   A[6]->key);
	cr_expect(4000   == A[7]->key, "Expected 4000 got %"PRId64,   A[7]->key);
}

Test(binary_heap, deletemin) {
	Node_t *val;

	val = BinaryHeapDeleteMin(heap);
	cr_expect(NULL == val, "Expected NULL");
	free(val);

	BinaryHeapInsert(heap, -8274);
	val = BinaryHeapDeleteMin(heap);
	cr_expect(-8274 == val->key, "Expected minimum to be -8274 got %"PRId64, val->key);
	free(val);

	BinaryHeapInsert(heap, -8274);
	BinaryHeapInsert(heap, 89474);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(-8274 == val->key, "Expected minimum to be -8274 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(89474 == val->key, "Expected minimum to be 89474 got %"PRId64, val->key);
	free(val);

	BinaryHeapInsert(heap, -8274);
	BinaryHeapInsert(heap, 89474);
	BinaryHeapInsert(heap, 327499);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(-8274 == val->key, "Expected minimum to be -8274 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(89474 == val->key, "Expected minimum to be 89474 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(327499 == val->key, "Expected minimum to be 327499 got %"PRId64, val->key);
	free(val);

	BinaryHeapInsert(heap, -8274);
	BinaryHeapInsert(heap, 89474);
	BinaryHeapInsert(heap, 327499);
	BinaryHeapInsert(heap, 0);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(-8274 == val->key, "Expected minimum to be -8274 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(0 == val->key, "Expected minimum to be 0 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(89474 == val->key, "Expected minimum to be 89474 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(327499 == val->key, "Expected minimum to be 327499 got %"PRId64, val->key);
	free(val);

	BinaryHeapInsert(heap, -8274);
	BinaryHeapInsert(heap, 89474);
	BinaryHeapInsert(heap, 327499);
	BinaryHeapInsert(heap, 0);
	BinaryHeapInsert(heap, -328);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(-8274 == val->key, "Expected minimum to be -8274 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(-328 == val->key, "Expected minimum to be -328 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(0 == val->key, "Expected minimum to be 0 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(89474 == val->key, "Expected minimum to be 89474 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(327499 == val->key, "Expected minimum to be 327499 got %"PRId64, val->key);
	free(val);

	BinaryHeapInsert(heap, -8274);
	BinaryHeapInsert(heap, 89474);
	BinaryHeapInsert(heap, 327499);
	BinaryHeapInsert(heap, 0);
	BinaryHeapInsert(heap, -328);
	BinaryHeapInsert(heap, -348023);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(-348023 == val->key, "Expected minimum to be -348023 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(-8274 == val->key, "Expected minimum to be -8274 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(-328 == val->key, "Expected minimum to be -328 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(0 == val->key, "Expected minimum to be 0 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(89474 == val->key, "Expected minimum to be 89474 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(327499 == val->key, "Expected minimum to be 327499 got %"PRId64, val->key);
	free(val);

	BinaryHeapInsert(heap, -8274);
	BinaryHeapInsert(heap, 89474);
	BinaryHeapInsert(heap, 327499);
	BinaryHeapInsert(heap, 0);
	BinaryHeapInsert(heap, -328);
	BinaryHeapInsert(heap, -348023);
	BinaryHeapInsert(heap, 1);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(-348023 == val->key, "Expected minimum to be -348023 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(-8274 == val->key, "Expected minimum to be -8274 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(-328 == val->key, "Expected minimum to be -328 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(0 == val->key, "Expected minimum to be 0 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(1 == val->key, "Expected minimum to be 1 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(89474 == val->key, "Expected minimum to be 89474 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(327499 == val->key, "Expected minimum to be 327499 got %"PRId64, val->key);
	free(val);

	BinaryHeapInsert(heap, -8274);
	BinaryHeapInsert(heap, 89474);
	BinaryHeapInsert(heap, 327499);
	BinaryHeapInsert(heap, 0);
	BinaryHeapInsert(heap, -328);
	BinaryHeapInsert(heap, -348023);
	BinaryHeapInsert(heap, 1);
	BinaryHeapInsert(heap, INT64_MAX);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(-348023 == val->key, "Expected minimum to be -348023 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(-8274 == val->key, "Expected minimum to be -8274 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(-328 == val->key, "Expected minimum to be -328 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(0 == val->key, "Expected minimum to be 0 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(1 == val->key, "Expected minimum to be 1 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(89474 == val->key, "Expected minimum to be 89474 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(327499 == val->key, "Expected minimum to be 327499 got %"PRId64, val->key);
	free(val);

	val = BinaryHeapDeleteMin(heap);
	cr_expect(INT64_MAX == val->key, "Expected minimum to be %"PRId64" got %"PRId64, INT64_MAX, val->key);
	free(val);
}

Test(binary_heap, delete) {
	Node_t **A;

	Node_t *node1, *node3;
	node1 = BinaryHeapInsert(heap, -328);
	BinaryHeapInsert(heap, 394);
	node3 = BinaryHeapInsert(heap, 294);
	BinaryHeapInsert(heap, -32489);

	BinaryHeapDelete(heap, node3);

	BinaryHeapInsert(heap, 38);
	BinaryHeapInsert(heap, 0);
	BinaryHeapInsert(heap, 4000);

	BinaryHeapDelete(heap, node1);

	BinaryHeapInsert(heap, 4000);

	A = heap->A;

	DrawBinaryHeap(heap, "deletebinary");

	cr_expect(-32489 == A[0]->key, "Expected -32489 got %"PRId64, A[0]->key);
	cr_expect(0      == A[1]->key, "Expected 0 got %"PRId64,      A[1]->key);
	cr_expect(394    == A[2]->key, "Expected 394 got %"PRId64,    A[2]->key);
	cr_expect(38     == A[3]->key, "Expected 38 got %"PRId64,     A[3]->key);
	cr_expect(4000   == A[4]->key, "Expected 4000 got %"PRId64,   A[4]->key);
	cr_expect(4000   == A[5]->key, "Expected 4000 got %"PRId64,   A[5]->key);
}

// TODO: Test INT64_MIN
Test(binary_heap, decreasekey) {
	Node_t **A;
	Node_t *node1, *node3, *node7;
	node1 = BinaryHeapInsert(heap, -328);
	BinaryHeapInsert(heap, 394);
	node3 = BinaryHeapInsert(heap, 294);
	BinaryHeapInsert(heap, -32489);

	BinaryHeapDelete(heap, node3);

	BinaryHeapInsert(heap, 38);
	BinaryHeapInsert(heap, 0);
	node7 = BinaryHeapInsert(heap, 4000);

	BinaryHeapDelete(heap, node1);

	BinaryHeapInsert(heap, 4000);

	BinaryHeapDecreaseKey(heap, 104001, node7);

	A = heap->A;

	cr_expect(-100001 == A[0]->key, "Expected 4000 got %"PRId64,   A[0]->key);
	cr_expect(-32489  == A[1]->key, "Expected -32489 got %"PRId64, A[1]->key);
	cr_expect(394     == A[2]->key, "Expected 394 got %"PRId64,    A[2]->key);
	cr_expect(38      == A[3]->key, "Expected 38 got %"PRId64,     A[3]->key);
	cr_expect(0       == A[4]->key, "Expected 0 got %"PRId64,      A[4]->key);
	cr_expect(4000    == A[5]->key, "Expected 4000 got %"PRId64,   A[5]->key);
}
