#define _DEFAULT_SOURCE
#include <stdlib.h>
#include <stdbool.h>
#include <stdio.h>
#include <string.h>
#include <inttypes.h>
#include <getopt.h>
#include <dirent.h>
#include <errno.h>

#include "xutil.h"
#include "node.h"
#include "fibonacciheap.h"
#include "binaryheap.h"
#include "matrix.h"
#include "heap.h"
#include "dijkstra.h"

#include <measure.h>

#ifdef COUNT_COMPARISONS
uint64_t bin_comps = 0;
uint64_t bin_insert_comps = 0;
uint64_t bin_deletemin_comps = 0;
uint64_t bin_decreasekey_comps = 0;

uint64_t fib_comps = 0;
uint64_t fib_insert_comps = 0;
uint64_t fib_deletemin_comps = 0;
uint64_t fib_decreasekey_comps = 0;
#endif

//static uint64_t comparisons;

heap_t fibonacci_heap = {
	.create        = (Create) MakeFibonacciHeap,
	.free          = (Free) FreeFibonacciHeap,
	.insert        = (Insert) FibonacciHeapInsert,
	.deletemin     = (DeleteMin) FibonacciHeapDeleteMin,
	.delete        = (Delete) FibonacciHeapDelete,
	.findmin       = (FindMin) FibonacciHeapFindMin,
	.decreasekey   = (DecreaseKey) FibonacciHeapDecreaseKey,
	.decreasekeyto = (DecreaseKeyTo) FibonacciHeapDecreaseKeyTo,
	.empty         = (IsEmpty) FibonacciHeapIsEmpty
};

heap_t binary_heap = {
	.create        = (Create) MakeBinaryHeap,
	.free          = (Free) FreeBinaryHeap,
	.insert        = (Insert) BinaryHeapInsert,
	.deletemin     = (DeleteMin) BinaryHeapDeleteMin,
	.delete        = (Delete) BinaryHeapDelete,
	.findmin       = (FindMin) BinaryHeapFindMin,
	.decreasekey   = (DecreaseKey) BinaryHeapDecreaseKey,
	.decreasekeyto = (DecreaseKeyTo) BinaryHeapDecreaseKeyTo,
	.empty         = (IsEmpty) BinaryHeapIsEmpty
};

struct heap_measure {
	char   *file;
	heap_t *heap;
};

void
printusage(char *argv[]) {
	fprintf(stderr, "Usage: %s \n"
			"\t[-f (Use Fibonacci Heap)]\n"
			"\t[-b (Use Binary Heap)]\n"
			"\t[-d (Run Dijkstra)]\n"
			"\t[-i (Input file)]\n"
			"\t[-p (Input folder)]\n"
			"\t[-r (Repetitions)]\n"
			"\t[-t (Test heaps against each other)]\n"
			, argv[0]);
}

static int
cmpstringp(const void *p1, const void *p2) {
	/* The actual arguments to this function are "pointers to
	 * pointers to char", but strcmp(3) arguments are "pointers
	 * to char", hence the following cast plus dereference */

	return strcmp(* (char * const *) p1, * (char * const *) p2);
}

void read_and_compare_heaps(char *filename, heap_t *heap1, heap_t *heap2) {
	FILE *fd;
	int res;
	int c = 0;
	int64_t i;
	uint64_t val, j, count = 0, size = 2, line = 0;
	size_t bufsize = 256;
	char *buf = malloc(bufsize);
	Node_t *res1, *res2;
	Node_t **inserted1 = xmalloc(size*sizeof(Node_t*));
	Node_t **inserted2 = xmalloc(size*sizeof(Node_t*));

	heap1->heap = heap1->create();
	heap2->heap = heap2->create();

	fd = fopen(filename, "r");
	if (fd == NULL) {
		xerror("Unable to open input file", __LINE__, __FILE__);
	}

	while ((c = getline(&buf, &bufsize, fd)) != -1) {
		line++;

		res = sscanf(buf, "Insert(%"PRId64")\n", &i);
		if (res == 1) {
			//INSERT
			
			if ( unlikely(count+1 > size) ) {
				size = 2*(count+1);
				inserted1 = xrealloc(inserted1, size*sizeof(Node_t *));
				inserted2 = xrealloc(inserted2, size*sizeof(Node_t *));
			}

			inserted1[count] = heap1->insert(heap1->heap, i);
			inserted2[count] = heap2->insert(heap2->heap, i);

			count++;
			continue;
		}

		res = strncmp(buf, "DeleteMin()\n", c);
		if (res == 0) {
			//DELETE-MIN
			res1 = heap1->deletemin(heap1->heap);
			res2 = heap2->deletemin(heap2->heap);
			if (res1->key != res2->key) {
				fprintf(stderr, "%"PRId64" != %"PRId64", line %"PRIu64"\n", res1->key, res2->key, line);
				xerror("Error in DeleteMin", __LINE__, __FILE__);
			}
			free(res1);
			free(res2);
			continue;
		}

		res = sscanf(buf, "Delete(%"PRIu64")\n", &j);
		if (res == 1) {
			//DELETE
			heap1->delete(heap1->heap, inserted1[j]);
			heap2->delete(heap2->heap, inserted2[j]);
			continue;
		}

		res = strncmp(buf, "FindMin()\n", c);
		if (res == 0) {
			//FIND-MIN
			res1 = heap1->findmin(heap1->heap);
			res2 = heap2->findmin(heap2->heap);
			if (res1->key != res2->key) {
				fprintf(stderr, "%"PRId64" != %"PRId64", line %"PRIu64"\n", res1->key, res2->key, line);
				xerror("Error in FindMin", __LINE__, __FILE__);
			}
			continue;
		}

		res = sscanf(buf, "DecreaseKey(%"PRIu64", %"PRIu64")\n", &val, &j);
		if (res == 2) {
			// DecreaseKey
			heap1->decreasekey(heap1->heap, val, inserted1[j]);
			heap2->decreasekey(heap2->heap, val, inserted2[j]);
			continue;
		}

		fprintf(stderr, "Line: %"PRIu64"\n", line);
		xerror("Unable to parse line of input", __LINE__, __FILE__);
	}

	heap1->free(heap1->heap);
	heap2->free(heap2->heap);

	free(inserted1);
	free(inserted2);

	fclose(fd);

	free(buf);
}

void read_and_measure_heap(struct heap_measure *restrict hm) {
	int res;
	int64_t i;
	uint64_t val, j, count = 0, size = 2;
	heap_t *heap = (heap_t *) hm->heap;
	char *file = hm->file;
	Node_t **inserted = xmalloc(size*sizeof(Node_t*));

	heap->heap = heap->create();

	bool DEBUG = false;

	int line = 0;
	(void) line;
	while (1) {
		if (DEBUG) {
			printf("Line: %d\n", ++line);
		}
		res = sscanf(file, "Insert(%"PRId64")", &i);
		if (res == 1) {
			//INSERT
			if (DEBUG) {
				fprintf(stdout, "Insert(%"PRIu64")\n", i);
			}
			
			if ( unlikely(count+1 > size) ) {
				size = 2*(count+1);
				inserted = xrealloc(inserted, size*sizeof(Node_t *));
			}

			inserted[count] = heap->insert(heap->heap, i);
			inserted[count]->v = count;
			count++;

			goto END;
		}

		char *DM =  "DeleteMin()";
		res = strncmp(file, DM, strlen(DM));
		if (res == 0) {
			//DELETE-MIN
			if (DEBUG) {
				fprintf(stdout, "DeleteMin()\n");
			}
			Node_t *in = heap->deletemin(heap->heap);
			if (DEBUG) {
				printf("Minimum: %lu\n", in->v);
				printf("Value: %lu\n", in->key);
			}
			free(in);

			goto END;
		}

		res = sscanf(file, "Delete(%"PRIu64")", &j);
		if (res == 1) {
			//DELETE
			if (DEBUG) {
				fprintf(stdout, "Delete(%"PRIu64")\n", i);
			}
			heap->delete(heap->heap, inserted[j]);

			goto END;
		}

		char *FM = "FindMin()";
		res = strncmp(file, FM, strlen(FM));
		if (res == 0) {
			//FIND-MIN
			if (DEBUG) {
				fprintf(stdout, "FindMin()\n");
			}
			heap->findmin(heap->heap);

			goto END;
		}

		res = sscanf(file, "DecreaseKey(%"PRIu64", %"PRIu64")", &val, &j);
		if (res == 2) {
			// DECREASE-KEY
			if (DEBUG) {
				fprintf(stdout, "DecreaseKey(%"PRIu64", %"PRIu64")\n", val, j);
			}
			heap->decreasekey(heap->heap, val, inserted[j]);

			goto END;
		}

		xerror("Unable to parse line of input", __LINE__, __FILE__);

END:
		while (*file != '\n') {
			file++;
		}

		file++;

		if (*file == '\0') {
			break;
		}
	}

	heap->free(heap->heap);

	free(inserted);
}


void run_dijkstra(char *desc, char *file, struct dijk_measure *d, int reps) {
	char buf[100];
	buf[0] = '\0';
	
	char *d1 = "Dijkstra ";
	char *d2 = "Dijkstra2 ";

	strncat(buf, d1, 99);
	strncat(buf,desc, 80);

	measure(
			buf,
			file,
			(testfunc)Dijkstra,
			(void *) d,
			reps
	);

	buf[0] = '\0';
	strncat(buf, d2, 99);
	strncat(buf,desc, 80);

	measure(
			buf,
			file,
			(testfunc)Dijkstra2,
			(void *) d,
			reps
	);
}

int
main (int argc, char**argv) {
	struct heap_measure *hm;
	struct dijk_measure *d;
	bool dijkstra = false;
	bool fibonacci = false;
	bool binary = false;
	bool test = false;
	const char *optstring = "tfbdp:i:r:";
	uint64_t REPS_MEASURE = 100;
	uint64_t i;
	int opt;

	DIR     *dir;
	struct   dirent *entry;
	matrix_t *matrix;
	char   **files = NULL;
	uint32_t filecount = 0;
	char    *path = NULL;
	char    *input = NULL;
	FILE    *fd;
	uint64_t filesize;

	while ((opt = getopt(argc, argv, optstring)) != -1) {
		switch(opt) {
			case 'f':
				fibonacci = true;
				break;
			case 'b':
				binary = true;
				break;
			case 'd':
				dijkstra = true;
				break;
			case 'p':
				path = strndup(optarg, 256);
				break;
			case 'i':
				input = strndup(optarg, 256);
				break;
			case 'r':
				REPS_MEASURE = strtol(optarg, NULL, 10);
				break;
			case 't':
				test = true;
				break;
			default:
				printusage(argv);
				exit(EXIT_FAILURE);
		}
	}

	// If neither input file or folder is given
	if ( input == NULL && path == NULL ) {
		printusage(argv);
		exit(EXIT_FAILURE);
	}

	// If no specific input file was given
	if (input == NULL) {
		filecount = xfilesdir(path);
		files = xcalloc(filecount, sizeof(char *));

		if ((dir = opendir(path)) == NULL) {
			xerror(strerror(errno), __LINE__, __FILE__);
		}

		i = 0;
		while ( (entry = readdir(dir)) != NULL ) {
			if (entry->d_type == DT_REG) { /* If the entry is a regular file */
				files[i] = xmalloc(sizeof(char)*(256+strlen(path)+1));
				memcpy(files[i], path, strlen(path));
				memcpy(&files[i][strlen(path)], "/", 1);
				memcpy(&files[i][strlen(path)+1], entry->d_name, 256);
				i++;
			}
		}

		qsort(files, filecount, sizeof(char *), cmpstringp);
	}

	if ( !fibonacci && !binary ) {
		binary = true;
		fibonacci = true;
	} 


	if ( test ) {
		if ( dijkstra ) {
			Node_t **nodes1, **nodes2, **nodes3, **nodes4;
			Node_t **res1, **res2, **res3, **res4;
			uint64_t j;
			d = xmalloc(sizeof(struct dijk_measure));

			d->result = true;
			d->source = 0;

			if ( NULL == input ) {
				for (i = 0; i < filecount; i++) {
					printf("Testing: %s\n", files[i]);

					matrix = read_graph_matrix(files[i]);

					d->matrix = matrix;

					d->heap = &binary_heap;
					res1 = Dijkstra(d);
					nodes1 = d->nodes;
					res2 = Dijkstra2(d);
					nodes2 = d->nodes;

					d->heap = &fibonacci_heap;
					res3 = Dijkstra(d);
					nodes3 = d->nodes;
					res4 = Dijkstra2(d);
					nodes4 = d->nodes;

					for (j = 0; j < matrix->n; j++) {
						if ( res1[j]->key != res2[j]->key ) {
							fprintf(stderr, "res1:%"PRId64" != res2:%"PRId64"\n", res1[j]->key, res2[j]->key);
							xerror("Expected result to be equal", __LINE__, __FILE__);
						}

						if ( res1[j]->key != res3[j]->key ) {
							fprintf(stderr, "res1:%"PRId64" != res3:%"PRId64"\n", res1[j]->key, res3[j]->key);
							xerror("Expected result to be equal", __LINE__, __FILE__);
						}

						if ( res1[j]->key != res4[j]->key ) {
							fprintf(stderr, "res1:%"PRId64" != res4:%"PRId64"\n", res1[j]->key, res4[j]->key);
							xerror("Expected result to be equal", __LINE__, __FILE__);
						}
					}

					for (j = 0; j < matrix->n; j++) {
						free(nodes1[j]);
						free(nodes2[j]);
						free(nodes3[j]);
						free(nodes4[j]);
					}
					free(nodes1);
					free(nodes2);
					free(nodes3);
					free(nodes4);

					free(res1);
					free(res2);
					free(res3);
					free(res4);

					matrix_destroy(matrix);
				}
			} else {
				printf("Testing: %s\n", input);
				matrix = read_graph_matrix(input);

				d->matrix = matrix;

				d->heap = &binary_heap;
#ifdef COUNT_COMPARISONS
				extern uint64_t fib_comps;
#endif
				res1 = Dijkstra(d);
#ifdef COUNT_COMPARISONS
				printf("Bin Dijkstra comparisons: %"PRIu64"\n", bin_comps);
				printf("Bin Insert comparisons: %"PRIu64"\n", bin_insert_comps);
				printf("Bin DeleteMin comparisons: %"PRIu64"\n", bin_deletemin_comps);
				printf("Bin DecreaseKey comparisons: %"PRIu64"\n", bin_decreasekey_comps);
				bin_comps = 0;
				bin_insert_comps = 0;
				bin_deletemin_comps = 0;
				bin_decreasekey_comps = 0;
#endif
				nodes1 = d->nodes;
				res2 = Dijkstra2(d);
#ifdef COUNT_COMPARISONS
				printf("Bin Dijkstra2 comparisons: %"PRIu64"\n", bin_comps);
				printf("Bin Insert comparisons: %"PRIu64"\n", bin_insert_comps);
				printf("Bin DeleteMin comparisons: %"PRIu64"\n", bin_deletemin_comps);
				printf("Bin DecreaseKey comparisons: %"PRIu64"\n", bin_decreasekey_comps);
				bin_comps = 0;
				bin_insert_comps = 0;
				bin_deletemin_comps = 0;
				bin_decreasekey_comps = 0;
#endif
				nodes2 = d->nodes;

				d->heap = &fibonacci_heap;
				res3 = Dijkstra(d);
#ifdef COUNT_COMPARISONS
				printf("Fib Dijkstra comparisons: %"PRIu64"\n", fib_comps);
				printf("Fib Insert comparisons: %"PRIu64"\n", fib_insert_comps);
				printf("Fib DeleteMin comparisons: %"PRIu64"\n", fib_deletemin_comps);
				printf("Fib DecreaseKey comparisons: %"PRIu64"\n", fib_decreasekey_comps);
				fib_comps = 0;
				fib_insert_comps = 0;
				fib_deletemin_comps = 0;
				fib_decreasekey_comps = 0;
#endif
				nodes3 = d->nodes;
				res4 = Dijkstra2(d);
#ifdef COUNT_COMPARISONS
				printf("Fib Dijkstra2 comparisons: %"PRIu64"\n", fib_comps);
				printf("Fib Insert comparisons: %"PRIu64"\n", fib_insert_comps);
				printf("Fib DeleteMin comparisons: %"PRIu64"\n", fib_deletemin_comps);
				printf("Fib DecreaseKey comparisons: %"PRIu64"\n", fib_decreasekey_comps);
				fib_comps = 0;
				fib_insert_comps = 0;
				fib_deletemin_comps = 0;
				fib_decreasekey_comps = 0;
#endif
				nodes4 = d->nodes;

				for (j = 0; j < matrix->n; j++) {
					if ( res1[j]->key != res2[j]->key ) {
						fprintf(stderr, "j: %"PRIu64", res1:%"PRId64" != res2:%"PRId64"\n", j, res1[j]->key, res2[j]->key);
						xerror("Expected result to be equal", __LINE__, __FILE__);
					}

					if ( res1[j]->key != res3[j]->key ) {
						fprintf(stderr, "j: %"PRIu64", res1:%"PRId64" != res3:%"PRId64"\n", j, res1[j]->key, res3[j]->key);
						xerror("Expected result to be equal", __LINE__, __FILE__);
					}

					if ( res1[j]->key != res4[j]->key ) {
						fprintf(stderr, "j: %"PRIu64", res1:%"PRId64" != res4:%"PRId64"\n", j, res1[j]->key, res4[j]->key);
						xerror("Expected result to be equal", __LINE__, __FILE__);
					}
				}

				for (j = 0; j < matrix->n; j++) {
					free(nodes1[j]);
					free(nodes2[j]);
					free(nodes3[j]);
					free(nodes4[j]);
				}

				free(nodes1);
				free(nodes2);
				free(nodes3);
				free(nodes4);

				free(res1);
				free(res2);
				free(res3);
				free(res4);

				matrix_destroy(matrix);
			}

			if ( NULL != d ) {
				free(d);
			}
		} else {
			if ( NULL == input ) {
				for (i = 0; i < filecount; i++) {
					printf("Testing: %s\n", files[i]);
					read_and_compare_heaps(files[i], &fibonacci_heap, &binary_heap);
				}
			} else {
				printf("Testing: %s\n", input);
				read_and_compare_heaps(input, &fibonacci_heap, &binary_heap);
			}
		}
		printf("\nSUCCESSFULLY TESTED\n");
	} else if ( dijkstra ) {
		if ( !measure_init() ) {
			xerror("Error initializing measure library", __LINE__, __FILE__);
		}

		d = xmalloc(sizeof(struct dijk_measure));

		/* DIJKSTRA */
		if ( NULL == input ) {
			for (i = 0; i < filecount; i++) {
				matrix = read_graph_matrix(files[i]);

				d->matrix = matrix;
				d->source = 0;
				d->result = false;

				if (fibonacci) {
					d->heap = &fibonacci_heap;
					run_dijkstra("Fibonacci", files[i], d, REPS_MEASURE);
				}

				if (binary) {
					d->heap = &binary_heap;
					run_dijkstra("Binary", files[i], d, REPS_MEASURE);
				}

				matrix_destroy(matrix);
			}
		} else {
			matrix = read_graph_matrix(input);

			d->matrix = matrix;
			d->source = 0;
			d->result = false;

			if (fibonacci) {
				d->heap = &fibonacci_heap;
				run_dijkstra("Fibonacci", input, d, REPS_MEASURE);
			}

			if (binary) {
				d->heap = &binary_heap;
				run_dijkstra("Binary", input, d, REPS_MEASURE);
			}

			matrix_destroy(matrix);
		}
		
		/* DIJKSTRA */

		if ( NULL != d ) {
			free(d);
		}

		measure_destroy();
	} else {
		if ( !measure_init() ) {
			xerror("Error initializing measure library", __LINE__, __FILE__);
		}

		hm = xmalloc(sizeof(struct heap_measure));

		if ( NULL == input ) {
			// We are looking in a folder.
			for (i = 0; i < filecount; i++) {
				filesize = xfilesize(files[i]);

				fd = fopen(files[i], "r");
				if (fd == NULL) {
					xerror("Unable to open input file\n", __LINE__, __FILE__);
				}

				hm->file = xmalloc( sizeof(char)*(filesize + 1));
				if ( filesize != fread(hm->file, 1, filesize, fd) ) {
					xerror("Didn't read enough\n", __LINE__, __FILE__);
				}

				hm->file[filesize] = '\0';

				if ( fibonacci ) {
					hm->heap     = &fibonacci_heap;

					measure(
							"Fibonacci Heap",
							files[i],
							(testfunc)read_and_measure_heap,
							(void *) hm,
							REPS_MEASURE
					);
				}

				if ( binary ) {
					hm->heap     = &binary_heap;

					measure(
							"Binary Heap",
							files[i],
							(testfunc)read_and_measure_heap,
							(void *) hm,
							REPS_MEASURE
					);
				}

				free(hm->file);
			}
		} else {
			filesize = xfilesize(input);

			fd = fopen(input, "r");
			if (fd == NULL) {
				xerror("Unable to open input file\n", __LINE__, __FILE__);
			}

			hm->file = xmalloc( sizeof(char)*(filesize + 1));
			if ( filesize != fread(hm->file, 1, filesize, fd) ) {
				xerror("Didn't read enough\n", __LINE__, __FILE__);
			}

			hm->file[filesize] = '\0';

			// We are running one file only
			if ( fibonacci ) {
				hm->heap = &fibonacci_heap;

				measure(
						"Fibonacci Heap",
						input,
						(testfunc)read_and_measure_heap,
						(void *) hm,
						REPS_MEASURE
				);
			}

			if ( binary ) {
				hm->heap = &binary_heap;

				measure(
						"Binary Heap",
						input,
						(testfunc)read_and_measure_heap,
						(void *) hm,
						REPS_MEASURE
				);
			}

			free(hm->file);
		}

		if ( NULL != hm ) {
			free(hm);
		}

		measure_destroy();
	}

	// Clean up memory usage
	if (files != NULL) {
		for (i = 0; i < filecount; i++) {
			if ( NULL != files[i] ) {
				free(files[i]);
			}
		}
		free(files);
	}

	if ( NULL != input ) {
		free(input);
	}

	if ( NULL != path ) {
		free(path);
	}

	return EXIT_SUCCESS;
}
