#define _DEFAULT_SOURCE

#include <stdlib.h>
#include <stdio.h>
#include <stdbool.h>
#include <string.h>
#include <inttypes.h>
#include <errno.h>
#include <math.h>
#include <assert.h>

#include "xutil.h"
#include "node.h"
#include "fibonacciheap.h"

static double logphi = 0.4812118250596034707556;
static FibonacciHeap_t meld_heap = { NULL, 0 };

FibonacciHeap_t *
MakeFibonacciHeap() {
	FibonacciHeap_t *heap = xmalloc( sizeof(FibonacciHeap_t) );

	heap->min = NULL;
	heap->n = 0;

	return heap;
}

void
recursivelyFreeLevel(Node_t *n) {
	Node_t *next, *temp;

	if ( NULL == n ) {
		return;
	}

	next = n;
	do {
		temp = next;
		recursivelyFreeLevel(next->child);
		next = next->nextSibling;
		if ( NULL != temp && temp != n ) {
			free(temp);
		}
	} while (next != n);

	free(n);
}

void
FreeFibonacciHeap(FibonacciHeap_t *h) {
	if ( NULL != h ) {
		if ( NULL != h->min ) {
			recursivelyFreeLevel(h->min);
		}
		free(h);
	}
}

/* Melds the h2 into h1 */
static inline void
FibonacciHeapMeld(FibonacciHeap_t *h1, FibonacciHeap_t *h2) {
	Node_t *n1, *n2, *nextSibling; 

	n1 = h1->min;
	n2 = h2->min;
	if ( likely(NULL != n1) ) {
		// Update pointers
		nextSibling = n1->nextSibling;
		n2->nextSibling = nextSibling;
		n2->prevSibling = n1;
		n1->nextSibling = n2;
		nextSibling->prevSibling = n2;

		// Update minimum node
		if ( n2->key < n1->key ) {
			h1->min = n2;
		}
	} else {
		h1->min = n2;
	}
	h1->n += h2->n;
}

/* Returns the node with the minimum element with a new child */
static inline Node_t *
FibonacciHeapLink(Node_t *x, Node_t *y) {
	Node_t *prevSibling, *nextSibling, *root; 

#ifdef COUNT_COMPARISONS
	fib_comps++;
	fib_deletemin_comps++;
#endif
	if ( x->key < y->key ) {
		// Update sibling list
		y->nextSibling->prevSibling = y->prevSibling;
		y->prevSibling->nextSibling = y->nextSibling;

		// Update child list
		if ( NULL != x->child ) {
			nextSibling = x->child;
			prevSibling = nextSibling->prevSibling;
			y->nextSibling = nextSibling;
			y->prevSibling = prevSibling;
			nextSibling->prevSibling = y;
			prevSibling->nextSibling = y;
		} else {
			y->nextSibling = y->prevSibling = y;
			x->child = y;
		}
		
		// Unmark root node
		if ( NULL == y->parent ) {
			y->marked = false;
		}

		// Update parent
		y->parent = x;	
		x->rank++;
		root = x;
	} else {
		// Update sibling list
		x->nextSibling->prevSibling = x->prevSibling;
		x->prevSibling->nextSibling = x->nextSibling;

		// Update child list
		if ( NULL != y->child ) {
			nextSibling = y->child;
			prevSibling = nextSibling->prevSibling;
			x->nextSibling = nextSibling;
			x->prevSibling = prevSibling;
			nextSibling->prevSibling = x;
			prevSibling->nextSibling = x;
		} else {
			x->nextSibling = x->prevSibling = x;
			y->child = x;
		}

		// Unmark root node
		if ( NULL == x->parent ) {
			x->marked = false;
		}

		// Update parent
		x->parent = y;	
		y->rank++;
		root = y;
	}

	return root;
}

bool FibonacciHeapIsEmpty(FibonacciHeap_t *h) {
	return (bool) h->n == 0;
}

Node_t * 
FibonacciHeapFindMin(FibonacciHeap_t *h) {
	if ( unlikely(NULL == h->min) ) {
		return NULL;
	}
	return h->min;
}

Node_t *
FibonacciHeapInsert(FibonacciHeap_t *h, int64_t i) {
	Node_t *x = MakeNode(i);

	// Create new heap
	FibonacciHeap_t *hi = &meld_heap;
	hi->min = x;
	hi->n = 1;

	// Meld old and new heap
	FibonacciHeapMeld(h, hi);

	#ifdef COUNT_COMPARISONS
		fib_comps++;
		fib_insert_comps++;
	#endif

	return x;
}

Node_t * 
FibonacciHeapDeleteMin(FibonacciHeap_t *h) {
	uint64_t i, logSize;
	int64_t min = INT64_MAX;
	Node_t *x, *prevSibling, *nextSibling, *child, *childPrev, *next, *root,
		   *ptr, *tmp;

	// Store pointers
	x = h->min;

	if ( unlikely(NULL == x) ) {
		return x;
	}

	prevSibling = x->prevSibling;
	nextSibling = x->nextSibling;
	child       = x->child;

	// If it was the only node
	if ( unlikely(NULL == child && nextSibling == prevSibling && x == nextSibling) ) {
		h->min = NULL;
		h->n = 0;
#ifdef ROOTLENGTH
		printf("l, %d\n", 1);
#endif
		return x;
	}

	root = child;

	// Remove the minimum element and concatenate linked lists.
	if ( NULL == child ) {
		nextSibling->prevSibling = prevSibling;
		prevSibling->nextSibling = nextSibling;
		root                     = nextSibling;
	} else {
		// Reset parent pointers as all nodes in child list will be roots
		next = child;
		do {
			next->parent = NULL;
			next = next->nextSibling;
		} while (next != child);

		// Likely that minimum node has siblings
		if ( likely(nextSibling != prevSibling || x != nextSibling) ) {
			childPrev                = child->prevSibling;

			childPrev->nextSibling   = nextSibling;
			child->prevSibling       = prevSibling;
			prevSibling->nextSibling = child;
			nextSibling->prevSibling = childPrev;
		}
	}

	h->n--;

	// From CLRS chapter 19.4
	logSize = (uint64_t)floor(log(h->n)/logphi)+1;

	// For any practical purpose
	assert(logSize <= 64);

	// Allocate rank index table
	Node_t *index[logSize];
	memset(index, '\0', logSize * sizeof(Node_t *));

	// Iterate through root list
	next = root;

	// Linking step
#ifdef ROOTLENGTH
	int length = 0;
#endif
	while ( next->parent == NULL && next != index[next->rank] ) {
#ifdef ROOTLENGTH
		length++;
#endif
		ptr = next;
		tmp = index[ptr->rank];
		next = ptr->nextSibling;

		while ( NULL != tmp ) {
			index[ptr->rank] = NULL;
			root = ptr = FibonacciHeapLink(ptr, tmp);
			tmp = index[ptr->rank];
		}

		index[ptr->rank] = ptr;
	}

#ifdef ROOTLENGTH
	printf("l, %d\n", length);
#endif

	// Find minimum root element and make parent pointer NULL.
	root = NULL;
	for (i = 0; i < logSize; i++) {
		next = index[i];
		if ( NULL != next ) { 
			next->parent = NULL;
#ifdef COUNT_COMPARISONS
			fib_comps++;
			fib_deletemin_comps++;
#endif
			if ( next->key <= min ) {
				min = next->key;
				root = next;
			}
		}
	}	

	h->min = root;

	return x;
}


void decreaseKeyFibonacci(FibonacciHeap_t *h, Node_t *x) {
	Node_t *parent, *nextSibling, *prevSibling;
	FibonacciHeap_t *hi;

DK_START:

	nextSibling = x->nextSibling;
	prevSibling = x->prevSibling;
	parent      = x->parent;

	// Delete from linked lists
	prevSibling->nextSibling = nextSibling;
	nextSibling->prevSibling = prevSibling;
	x->prevSibling           = x;
	x->nextSibling           = x;

	// Create new heap
	hi = &meld_heap;
	hi->min = x;
	hi->n = 0;

#ifdef COUNT_COMPARISONS
	fib_comps++;
	fib_decreasekey_comps++;
#endif

	// Meld old and new heap
	FibonacciHeapMeld(h, hi);

	// Cut edge from x to parent
	if ( likely(NULL != parent) ) {
		if ( parent->child == x ) {
			if ( nextSibling != x ) { 
				parent->child = nextSibling;
			} else {
				parent->child = NULL;
			}
		}

		parent->rank--;

		x->parent = NULL;

		// Parent is not root node
		if ( NULL != parent->parent ) {
			// If parent is cut perform CASCADING CUT i.e. remove parent from
			// its tree as well. This can be emulated as decreasing the key of
			// the parent.
			if ( parent->marked ) {
				x = parent;
				goto DK_START;
			} else {
				parent->marked = true;
			}
		}
	}
}

static inline void 
cascadingCut(FibonacciHeap_t *h, Node_t *x, Node_t *nextSibling, Node_t *parent) {
	if ( parent->child == x ) {
		if ( nextSibling != x ) { 
			parent->child = nextSibling;
		} else {
			parent->child = NULL;
		}
	}

	parent->rank--;

	x->parent = NULL;

	// Parent is not root node
	if ( NULL != parent->parent ) {
		// If parent is cut perform CASCADING CUT i.e. remove parent from
		// its tree as well. This can be emulated as decreasing the key of
		// the parent.
		if ( parent->marked ) {
			decreaseKeyFibonacci(h, parent);
		} else {
			parent->marked = true;
		}
	}
}

void
FibonacciHeapDecreaseKeyTo(FibonacciHeap_t *h, int64_t val, Node_t *x) {
#ifdef COUNT_COMPARISONS
	fib_comps++;
	fib_decreasekey_comps++;
#endif
	if ( val < x->key ) {
		x->key = val;

#ifdef COUNT_COMPARISONS
		fib_comps++;
		fib_decreasekey_comps++;
#endif
		if ( x->parent == NULL ) {
			if (x->key < h->min->key) {
				h->min = x;
			}
		} else if (x->key < x->parent->key) {
			decreaseKeyFibonacci(h, x);
		}
	}
}

void
FibonacciHeapDecreaseKey(FibonacciHeap_t *h, uint64_t val, Node_t *x) {
	if ( unlikely( (x->key < 0 && (val >= INT64_MAX-1 || (uint64_t)(val - x->key) > INT64_MAX-1)) || 
				   (x->key >= 0 && val > (uint64_t)x->key && (uint64_t)(val - x->key) > INT64_MAX-1) ) ) {
		x->key = INT64_MIN;
	} else {
		x->key -= val;
	}

#ifdef COUNT_COMPARISONS
	fib_comps++;
	fib_decreasekey_comps++;
#endif
	if ( x->parent == NULL ) {
		if (x->key < h->min->key) {
			h->min = x;
		}
		return;
	} else if ( x->key < x->parent->key ) {
		decreaseKeyFibonacci(h, x);
	}
}

void 
FibonacciHeapDelete(FibonacciHeap_t *h, Node_t *x) {
	Node_t *prevSibling, *child, *childPrev, *root, *next, *parent, *nextSibling;

	root = h->min;

	if ( unlikely(x == root) ) {
		free(FibonacciHeapDeleteMin(h));
		return;
	}

	nextSibling = x->nextSibling;
	prevSibling = x->prevSibling;

	// Update successor list
	nextSibling->prevSibling = prevSibling;
	prevSibling->nextSibling = nextSibling;

	child = x->child;
	if ( NULL != child ) {
		// Reset parent pointers as all nodes in child list will be roots
		next = child;
		while ( NULL != next->parent ) {
			next->parent = NULL;
			next = next->nextSibling;
		}
		
		prevSibling = root->prevSibling;
		childPrev = child->prevSibling;

		childPrev->nextSibling = root;
		child->prevSibling = prevSibling;

		prevSibling->nextSibling = child;
		root->prevSibling = childPrev;
	}

	parent = x->parent;

	if ( likely(NULL != parent) ) {
		cascadingCut(h, x, nextSibling, parent);
	}

	h->n--;

	free(x);
}
