#define _GNU_SOURCE

#include "streams.h"
#include "stream_common.h"
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <stdint.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdbool.h>

#include <assert.h>

typedef struct input {
	FILE *restrict fd;
	uint64_t length;
	int B;
	int dir;
	uint32_t *restrict buf;
	bool free_buffer;
} input_stream;

typedef struct output {
	FILE *restrict fd;
	int B;
	uint32_t *restrict buf;
	bool free_buffer;
} output_stream;

input_stream **in_list;
int in_len = 0;

output_stream **out_list;
int out_len = 0;

/*
 * INPUT STREAM OPERATIONS
 */

__attribute__ ((visibility("default")))
int fi_open(const char *restrict f, uint64_t start, uint64_t length, int B, int dir) {
	int id = 0;
	int res;
	struct stat stat_buf;

	id = stream_find_struct_storage((uintptr_t **)&in_list, &in_len);

	input_stream *restrict s =  xmalloc(sizeof(input_stream));
	in_list[id] = s;

	res = stat(f, &stat_buf);
	if (res == -1) {
		switch errno {
			case ENOENT:
				s->fd = fopen(f, "w+");
				break;
			default:
				printf("Unable to open output: %s\n", strerror(errno));
				exit(EXIT_FAILURE);
		}
	} else {
		s->fd = fopen(f, "r+");
	}

	if(s->fd == 0) {
		printf("Error in fi_open: '%s'\n", strerror(errno));
		exit(1);
	}

	s->B = B;
	s->dir = dir;
	s->buf = NULL;
	s->length = length;
	s->free_buffer = false;

	if (dir == STREAM_BACKWARDS) {
		start -= 1;
	}

	fseek(s->fd, sizeof(uint32_t) * start, SEEK_SET);

	return id;
}

__attribute__ ((visibility("default")))
void fi_setbuffer(int id, int B, uint32_t *restrict buf) {
	input_stream *restrict i = in_list[id];

	if(setvbuf(i->fd, (char *)buf, _IOFBF, B*sizeof(uint32_t)) != 0) {
		fprintf(stderr, "Unable to update fi buffer: %s\n", strerror(errno));
		exit(1);
	}
}

__attribute__ ((visibility("default")))
uint32_t fi_next(int id) {
	int32_t res;
	int r;
	input_stream *restrict i = in_list[id];

	assert(i->dir == STREAM_FORWARDS);

	if (i->B > 0 && i->buf == NULL) {
		i->free_buffer = true;
		i->buf = xcalloc(sizeof(uint32_t), i->B);
		if(setvbuf(i->fd, (char *)i->buf, _IOFBF, i->B*sizeof(uint32_t)) != 0) {
			fprintf(stderr, "Unable to set fi buffer: %s\n", strerror(errno));
			exit(1);
		}
	}

	r = fread_unlocked(&res, 1, sizeof(uint32_t), i->fd);
	(void) r;

	if (ferror(i->fd) != 0 && !feof(i->fd)) {
		errorAndDie(3);
	}

	i->length--;

	return res;
}

__attribute__ ((visibility("default")))
uint32_t fi_last(int id) {
	int32_t res;
	int r;
	input_stream *restrict i = in_list[id];

	assert(i->dir == STREAM_BACKWARDS);

	if (i->B > 0 && i->buf == NULL) {
		i->free_buffer = true;
		i->buf = xcalloc(sizeof(uint32_t), i->B);
		if(setvbuf(i->fd, (char *)i->buf, _IOFBF, i->B*sizeof(uint32_t)) != 0) {
			fprintf(stderr, "Unable to set fi buffer: %s\n", strerror(errno));
			exit(1);
		}
	}

	r = fread_unlocked(&res, 1, sizeof(uint32_t), i->fd);
	(void) r;

	if (ferror(i->fd) != 0 && !feof(i->fd)) {
		errorAndDie(10);
	}

	i->length--;

	if (i->length != 0) {
		if (fseek(i->fd, -2*sizeof(uint32_t), SEEK_CUR) == -1) {
			fprintf(stderr, "Error seeking in fi_last: %s\n", strerror(errno));
			exit(1);
		}
	}

	return res;
}

__attribute__ ((visibility("default")))
int fi_eos(int id) {
	input_stream *restrict i = in_list[id];
	if (i->length == 0) {
		return 1;
	}

	return 0;
}

__attribute__ ((visibility("default")))
void fi_close(int id) {
	input_stream *restrict i = in_list[id];
	fclose(i->fd);

	if (i->free_buffer && i->buf != NULL) {
		free(i->buf);
	}

	free(i);

	in_list[id] = 0;
}

__attribute__ ((visibility("default")))
void fi_seek(int id, long offset, int length) {
	input_stream *restrict i = in_list[id];

	i->length = length;

	if (i->dir == STREAM_BACKWARDS) {
		offset -= 1;
	}

	if(fseek(i->fd, sizeof(uint32_t) * offset, SEEK_SET) == -1) {
		fprintf(stderr, "Unable to fi seek: %s\n", strerror(errno));
		exit(1);
	}
}

/*
 * OUTPUT STREAM OPERATIONS
 */

__attribute__ ((visibility("default")))
int fo_create(const char *restrict f, uint64_t start, int B, int dir) {
	int od = 0;
	int res;
	struct stat stat_buf;

	od = stream_find_struct_storage((uintptr_t **)&out_list, &out_len);

	output_stream *restrict o = xmalloc(sizeof(output_stream));
	out_list[od] = o;

	res = stat(f, &stat_buf);
	if (res == -1) {
		switch errno {
			case ENOENT:
				o->fd = fopen(f, "w+");
				break;
			default:
				printf("Unable to open output: %s\n", strerror(errno));
				exit(EXIT_FAILURE);
		}
	} else {
		o->fd = fopen(f, "r+");
	}

	if(o->fd == 0) {
		printf("Error in fo_create: '%s'\n", strerror(errno));
		exit(1);
	}

	o->B = B;
	o->buf = NULL;
	o->free_buffer = false;

	fseek(o->fd, start*sizeof(uint32_t), SEEK_SET);

	return od;
}

__attribute__ ((visibility("default")))
void fo_setbuffer(int od, int B, uint32_t *buf) {
	output_stream *restrict o = out_list[od];

	if(setvbuf(o->fd, (char *)buf, _IOFBF, B*sizeof(uint32_t)) != 0) {
		fprintf(stderr, "Unable to update fo buffer: %s\n", strerror(errno));
		exit(1);
	}
}

__attribute__ ((visibility("default")))
void fo_write(int od, uint32_t v) {
	output_stream *restrict o = out_list[od];
	int r;

	if (o->B > 0 && o->buf == NULL) {
		o->free_buffer = true;
		o->buf = xcalloc(sizeof(uint32_t), o->B);
		if(setvbuf(o->fd, (char *)o->buf, _IOFBF, o->B*sizeof(uint32_t)) != 0) {
			fprintf(stderr, "Unable to set fo buffer: %s\n", strerror(errno));
			exit(1);
		}
	}

	r = fwrite_unlocked(&v, 1, sizeof(uint32_t), o->fd);
	(void) r;

	if(ferror(o->fd) != 0) {
		errorAndDie(4);
	}
}

__attribute__ ((visibility("default")))
void fo_prepend(int od, uint32_t v) {
	int r;
	output_stream *restrict o = out_list[od];

	if (o->B > 0 && o->buf == NULL) {
		o->free_buffer = true;
		o->buf = xcalloc(sizeof(uint32_t), o->B);
		if(setvbuf(o->fd, (char *)o->buf, _IOFBF, o->B*sizeof(uint32_t)) != 0) {
			fprintf(stderr, "Unable to set fo buffer: %s\n", strerror(errno));
			exit(EXIT_FAILURE);
		}
	}

	if (fseek(o->fd, -sizeof(uint32_t), SEEK_CUR) == -1) {
			fprintf(stderr, "Unable to seek in fo prepend 1: %s\n", strerror(errno));
			exit(EXIT_FAILURE);
	}

	r = fwrite_unlocked(&v, 1, sizeof(uint32_t), o->fd);
	(void) r;

	if (fseek(o->fd, -sizeof(uint32_t), SEEK_CUR) == -1) {
			fprintf(stderr, "Unable to seek in fo prepend 2: %s\n", strerror(errno));
			exit(EXIT_FAILURE);
	}

	if(ferror(o->fd) != 0 && !feof(o->fd)) {
		errorAndDie(4);
	}
}

__attribute__ ((visibility("default")))
void fo_close(int od) {
	output_stream *restrict o = out_list[od];
	fflush(o->fd);
	fclose(o->fd);

	if (o->free_buffer && o->buf != NULL) {
		free(o->buf);
	}

	free(o);

	out_list[od] = 0;
}

__attribute__ ((visibility("default")))
void fo_seek(int od, long offset, int length) {
	output_stream *restrict o = out_list[od];
	(void)length;

	if(fseek(o->fd, offset*sizeof(uint32_t), SEEK_SET) == -1) {
		fprintf(stderr, "Unable to fo seek: %s\n", strerror(errno));
		exit(1);
	}
}
