#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>

// For stat()
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "utilities.h"

/**
 * This function is used when an error occurs and we need to shut the program 
 * down.
 */
void error(char *msg, int line, char *file) {
	fprintf(stderr, "Error %d@%s: %s\n", line, file, msg);
	exit(EXIT_FAILURE);
}

uint64_t filesize(char *file) {
	struct stat buf;
	int r = stat(file, &buf);
	if (r == -1) {
		fprintf(stderr, "Cannot stat '%s': '%s'\n", file, strerror(errno));
		exit(EXIT_FAILURE);
	}

	return buf.st_size;
}

void *xcalloc(size_t nmemb, size_t size) {
	void *p = calloc(nmemb, size);

	if (p == NULL) {
		fprintf(stderr, "Unable to c-allocate memory.");
		exit(EXIT_FAILURE);
	}

	return p;
}

void *xmalloc(size_t size) {
	void *p = malloc(size);

	if (p == NULL) {
		fprintf(stderr, "Unable to m-allocate memory.");
		exit(EXIT_FAILURE);
	}

	return p;
}

void *xrealloc(void *ptr, size_t size) {
	void *p = realloc(ptr, size);

	if (p == NULL) {
		fprintf(stderr, "Unable to re-allocate memory.");
		exit(EXIT_FAILURE);
	}

	return p;
}

void removefile(char *f) {
	struct stat buf;
	int r = stat(f, &buf);
	if (r == -1) {
		return;
	}

	unlink(f);
}


void prepend_output_stream_error (int od, uint32_t v) {
	(void) od;
	(void) v;
	error("Stream type doesn't support prepend", __LINE__, __FILE__);
}

uint32_t last_input_stream_error (int id) {
	(void) id;
	error("Stream type doesn't support last", __LINE__, __FILE__);
	return 0;
}

void seek_input_stream_error (int id, long offset, int length) {
	(void) id;
	(void) offset;
	(void) length;
	error("Stream type doesn't support input seek", __LINE__, __FILE__);
}

void setbuffer_stream_error (int od, int B, uint32_t *buf) {
	extern int STREAM_TYPE;

	(void) od;
	(void) B;
	(void) buf;

	switch(STREAM_TYPE) {
		case 4:
			return;
		default:
			error("Stream type doesn't support setbuffer", __LINE__, __FILE__);
	}
}

int open_input_stream(char *f, uint64_t start, uint64_t length, int B, 
		int flag) {
	extern int STREAM_TYPE;
	int fd;

	switch(STREAM_TYPE) {
	case 1:
		fd = i_open(f, start, length);
		break;

	case 2:
		fd = fi_open(f, start, length, B, flag);
		break;

	case 3:
		fd = bi_open(f, start, length, B, flag);
		break;

	case 4:
		fd = mi_open(f, start, length, B, flag);
		break;

	default:
		error("Invalid stream type", __LINE__, __FILE__);
		break;
	}

	return fd;
}

int open_output_stream(char *f, uint64_t start, uint64_t length, int B, 
		int flag) {
	extern int STREAM_TYPE;
	int fd;

	switch(STREAM_TYPE) {
	case 1:
		fd = o_create(f, start);
		break;

	case 2:
		fd = fo_create(f, start, B, flag);
		break;

	case 3:
		fd = bo_create(f, start, B, flag);
		break;

	case 4:
		fd = mo_create(f, start, length, B, flag);
		break;

	default:
		error("Invalid stream type", __LINE__, __FILE__);
		break;
	}

	return fd;
}

void seek_output_stream(int od, long offset, int length) {
	extern int STREAM_TYPE;

	switch(STREAM_TYPE) {
	case 1:
		error("Stream type doesn't support output seek", __LINE__, __FILE__);
		break;

	case 2:
		fo_seek(od, offset, length);
		break;

	case 3:
		bo_seek(od, offset);
		break;

	case 4:
		mo_seek(od, offset, length);
		break;

	default:
		error("Invalid stream type", __LINE__, __FILE__);
		break;
	}
}

void init_input_stream(in_stream *in) {
	extern int STREAM_TYPE;

	switch(STREAM_TYPE) {
	case 1:
		in->next = i_next;
		in->last = last_input_stream_error;
		in->eos = i_eos;
		in->close = i_close;
		in->seek = seek_input_stream_error;
		in->setbuffer = setbuffer_stream_error;
		in->open = open_input_stream; 
		break;

	case 2:
		in->next = fi_next;
		in->last = fi_last;
		in->eos = fi_eos;
		in->close = fi_close;
		in->seek = fi_seek;
		in->setbuffer = fi_setbuffer;
		in->open = open_input_stream;
		break;

	case 3:
		in->next = bi_next;
		in->last = bi_last;
		in->eos = bi_eos;
		in->close = bi_close;
		in->seek = bi_seek;
		in->setbuffer = bi_setbuffer;
		in->open = open_input_stream;
		break;

	case 4:
		in->next = mi_next;
		in->last = mi_last;
		in->eos = mi_eos;
		in->close = mi_close;
		in->seek = mi_seek;
		in->setbuffer = setbuffer_stream_error;
		in->open = open_input_stream;
		break;

	default:
		error("Invalid stream type", __LINE__, __FILE__);
		break;
	}
}

void init_output_stream(out_stream *out) {
	extern int STREAM_TYPE;

	switch(STREAM_TYPE) {
	case 1:
		out->write = o_write;
		out->close = o_close;
		out->seek = seek_output_stream;
		out->setbuffer = setbuffer_stream_error;
		out->prepend = prepend_output_stream_error; 
		out->create = open_output_stream;
		break;

	case 2:
		out->write = fo_write;
		out->close = fo_close;
		out->seek = fo_seek;
		out->setbuffer = fo_setbuffer;
		out->prepend = fo_prepend; 
		out->create = open_output_stream;
		break;

	case 3:
		out->write = bo_write;
		out->close = bo_close;
		out->seek = seek_output_stream;
		out->setbuffer = bo_setbuffer;
		out->prepend = bo_prepend; 
		out->create = open_output_stream;
		break;

	case 4:
		out->write = mo_write;
		out->close = mo_close;
		out->seek = mo_seek;
		out->setbuffer = setbuffer_stream_error;
		out->prepend = mo_prepend; 
		out->create = open_output_stream;
		break;

	default:
		error("Invalid stream type", __LINE__, __FILE__);
		break;
	}
}
