#define _DEFAULT_SOURCE
#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <inttypes.h>
#include <streams.h>
#include <string.h>
#include <getopt.h>
#include <assert.h>

#include <sys/resource.h>
#include <time.h>
#include <stdbool.h>

#include "utilities.h"
#include "heap.h"

int STREAM_TYPE = 3;
int BUFFER = 0;
int FANOUT = 0;
int PAGE_SIZE = 0;

// Default memory we save for the stack, 8MiB.
static uint64_t stack_reserved = 8388608;

void printusage(char *argv[]) {
	fprintf(stderr, "Usage: %s \n"
			"\t[-M memory size]\n"
			"\t[-m fanout]\n"
			"\t[-B buffer size]\n"
			"\t[-P page size]\n"
			"\t[-S stream implementation (1-4 = A-D)]\n"
			"\t[-i inputfile]\n"
			"\t[-r] print rusage when done\n"
			, argv[0]);
}

static inline unsigned int get_parent(unsigned int k, unsigned int m) {
	if (k == 0) {
		error("Trying to find parent of the root node.", __LINE__, __FILE__);
	}
	return (k-((k-1)%m)+1)/m;
}

int check_consistency(external_heap_t *h) {
	uint64_t biggest[h->last_leaf +1];
	uint64_t smallest[h->last_leaf +1];
	uint64_t wrong_node_order[h->last_leaf +1];
	uint64_t wrong_priority[h->last_leaf +1];
	uint64_t count[h->last_leaf +1];
	uint64_t has_errors = 0;
	uint64_t i, new = 0, prev;
	uint64_t size = 0;
	int fd;


	for (i = 0; i <= h->last_leaf; i++) {
		if (i == 0 && get_node_length(h, i) == 0) {
			break;
		}

		biggest[i] = 0;
		wrong_node_order[i] = 0;
		wrong_priority[i] = 0;
		size += get_node_length(h, i);

		char *filename = get_file_from_node(i);
		int offset = get_file_offset_from_node(h, i) + h->node_header_size;

		fd = h->in->open(filename, offset, get_node_length(h, i), BUFFER,
				STREAM_FORWARDS);

		prev = h->in->next(fd);
		smallest[i] = prev;

		count[i] = 0;
		while (!h->in->eos(fd)) {
			new = h->in->next(fd);

			if (prev < new || wrong_node_order[i] > 0) {
				has_errors++;
				wrong_node_order[i]++;
			} else {
				prev = new;
			}

			count[i]++;
		}
		free(filename);
		h->in->close(fd);

		biggest[i] = new;
		if (i > 0 && biggest[get_parent(i, h->m)] > new) {
			has_errors++;
			wrong_priority[i]++;
		}
	}

	assert(size+h->i_buffer->count == h->count);

	if (!has_errors) {
		return 1;
	}

	printf("############################################################################################################################################################\n");
	printf("############################################################################################################################################################\n");
	printf("\n");
	printf("Inconsistent heap!! %" PRIu64 " errors!\n", has_errors);
	printf("\n");
	printf("               i              Parent               count         Wrong order      Wrong priority        Self biggest       Self smallest      Parent biggest\n");
	printf("------------------------------------------------------------------------------------------------------------------------------------------------------------\n");

	for (i = 0; i <= h->last_leaf; i++) {
		if (wrong_node_order[i] + wrong_priority[i] == 0) {
			continue;
		}

		if (i == 0) {
			printf("%16" PRIu64 "                   0    %16" PRIu64 "    %16" PRIu64 "    ", i, wrong_node_order[i], wrong_priority[i]);
		} else {
			printf("%16" PRIu64 "    %16u    %16" PRIu64 "    %16" PRIu64 "    %16" PRIu64 "    ", i, get_parent(i, h->m), count[i], wrong_node_order[i], wrong_priority[i]);
		}

		if (i == 0) {
			printf("%16" PRIu64 "    %16" PRIu64 "\n", biggest[i], smallest[i]);
		} else {
			printf("%16" PRIu64 "    %16" PRIu64 "    %16" PRIu64 "\n", biggest[i], smallest[i], biggest[i-1]);
		}
	}

	return 0;
}

int main(int argc, char *argv[]) {
	char *inputfile = strndup("TEST1", 5);
	int opt;
	int fd;
	uint64_t N;
	uint64_t i;
	in_stream in;
	out_stream out;

	int total_ns;
	int total_s;
	struct rusage usage;
	struct timespec begin;
	struct timespec end;
	bool PRINT_RUSAGE = false;

	// Default memory size 104857600 = 100MiB.
	uint64_t M = 104857600;


	// Used to hold the previous value of the heap
	uint32_t prev;
	(void) prev;

	// Used to hold the newest value of the heap
	uint32_t new;

	while ((opt = getopt(argc, argv, "M:m:B:P:S:i:r")) != -1) {
		switch (opt) {
			case 'M':
				M = strtoull(optarg, NULL, 10);
				break;
			case 'm':
				FANOUT = strtoull(optarg, NULL, 10);
				break;
			case 'B':
				BUFFER = strtol(optarg, NULL, 10);
				break;
			case 'P':
				PAGE_SIZE = strtol(optarg, NULL, 10);
				break;
			case 'S':
				STREAM_TYPE = strtol(optarg, NULL, 10);
				if(STREAM_TYPE > 4 || STREAM_TYPE < 1) {
					printusage(argv);
					exit(EXIT_FAILURE);
				}
				break;
			case 'i':
				free(inputfile);
				inputfile = strndup(optarg, 200);
				break;
			case 'r':
				PRINT_RUSAGE = true;
				break;
			default:
				printusage(argv);
				exit(EXIT_FAILURE);
		}
	}

	clock_gettime(CLOCK_REALTIME, &begin);

	N = filesize(inputfile)/sizeof(uint32_t);

	init_input_stream(&in);
	init_output_stream(&out);

	external_heap_t *h = heap_init(stack_reserved, M, &in, &out);

	fd = in.open(inputfile, 0, N, BUFFER, STREAM_FORWARDS);
	free(inputfile);
	while ( !in.eos(fd) ) {
		heap_insert(h, in.next(fd));
	}
	in.close(fd);

	assert( 1 == check_consistency(h) );

	prev = 0;

	for (i = 0; i < N; i++) {
		new = heap_delete_min(h);

		assert( prev <= new );

		prev = new;
	}

	heap_destroy(h);

	getrusage(RUSAGE_SELF, &usage);

	clock_gettime(CLOCK_REALTIME, &end);
	if (begin.tv_nsec <= end.tv_nsec) {
		total_ns = end.tv_nsec - begin.tv_nsec;
		total_s  = end.tv_sec  - begin.tv_sec;
	} else {
		total_ns = end.tv_nsec + (1e9 - begin.tv_nsec);
		total_s  = end.tv_sec - begin.tv_sec - 1;
	}

	if (PRINT_RUSAGE) {
		printf(
				"%d,%d,%ld,%ld,%ld,%ld,%ld,%ld,%ld,%ld,%ld,%ld,%ld,%ld,%ld,%ld\n",
				/* Total realworld seconds/nsec used */
				total_s,
				total_ns,
				/* Total amount of CPU user time used.  */
				usage.ru_utime.tv_sec,   // Seconds
				usage.ru_utime.tv_usec,  // Microseconds
				/* Total amount of CPU system time used.  */
				usage.ru_stime.tv_sec,   // Seconds
				usage.ru_stime.tv_usec,  // Microseconds
				/* Maximum resident set size (in kilobytes).  */
				usage.ru_maxrss,
				/* Amount of data segment memory used (kilobyte-seconds).  */
				usage.ru_idrss,
				/* Amount of stack memory used (kilobyte-seconds).  */
				usage.ru_isrss,
				/* Number of soft page faults (i.e. those serviced by reclaiming
				   a page from the list of pages awaiting reallocation.  */
				usage.ru_minflt,
				/* Number of hard page faults (i.e. those that required I/O).  */
				usage.ru_majflt,
				/* Number of times a process was swapped out of physical memory. */
				usage.ru_nswap,
				/* Number of input operations via the file system.  Note: This
				   and `ru_oublock' do not include operations with the cache.  */
				usage.ru_inblock,
				/* Number of output operations via the file system.  */
				usage.ru_oublock,
				/* Number of voluntary context switches, i.e. because the process
				   gave up the process before it had to (usually to wait for some
				   resource to be available).  */
				usage.ru_nvcsw,
				/* Number of involuntary context switches, i.e. a higher priority
				 * process became runnable or the current process used up its time
				 * slice. */
				usage.ru_nivcsw
				);
	}

	return EXIT_SUCCESS;
}
