#include <stdlib.h>
#include <stdint.h>
#include <stdio.h>
#include <inttypes.h>
#include <math.h>

#include <assert.h>

#include <streams.h>

#include "heap.h"
#include "utilities.h"
#include "hsort.h"
#include "dmerge.h"

#define ROOT_NODE 0

/* Ideas:
 * 1. Use delayed (cached) writes of length of nodes.
 * 2. When opening streams, let them use buffers supplied instead.
 * 3. Test speed of inlined functions?
 * 4. Handle when we bubble up to root node.
 * ..
 * n. ...
 */

extern int BUFFER;

unsigned int F = 0;

/**
 * This function is given as a function pointer to the internat heap structure
 * and should sort the heap as a min-heap.
 */
int internal_min_heap_comparer(uint32_t a, uint32_t b) {
	return a < b;
}

/**
 * This function calculates the fanout from the available memory and page size.
 *
 * M = 2mP + 2P + m* + 3*
 */
unsigned int get_fanout(uint64_t M, unsigned int P) {
	extern int FANOUT;
	size_t ptr_size = sizeof(el_t);
	if (FANOUT != 0) {
		return FANOUT;
	}
	return (unsigned int)ceil(
			((double)(-3*ptr_size+M-2*P))/((double)(ptr_size+2*P))
	);
}

/**
 * P = (M-(3+m) *)/(2 (1+m))
 */
unsigned int get_pagesize(uint64_t M, unsigned int m) {
	size_t ptr_size = sizeof(el_t);
	return (unsigned int)ceil(
			(M-(3+m)*ptr_size)/(2*(1+m))
	);
}

static inline uint64_t get_nodes_per_file(uint64_t M, unsigned int P, uint64_t m) {
	// Is right below 1 GiB ~ 22*45
	return 22;
}

/**
 * Function that returns the index of the parent of node k. 
 */
static inline unsigned int get_parent(unsigned int k, unsigned int m) {
	if (k == ROOT_NODE) {
		error("Trying to find parent of the root node.", __LINE__, __FILE__);
	}
	return (k-((k-1)%m)+1)/m;
}

/**
 * Function that returns the index of the first child.
 */
static inline unsigned int get_first_child(unsigned int k, unsigned int m) {
	return m*k+1;
}

/**
 * Calculated the number of children of a node.
 */
unsigned int get_number_of_children(external_heap_t *h, unsigned int k) {
	if (h->last_leaf == ROOT_NODE) {
		return 0;
	}

	unsigned int first_child = get_first_child(k, h->m);
	if (first_child > h->last_leaf) {
		return 0;
	}

	unsigned int first_to_last = h->last_leaf - first_child +1;
	if (first_to_last >= h->m) {
		return h->m;
	}

	return first_to_last;
}

/**
 * Finds the sibling index, starting from 0.
 */
static inline unsigned int get_sibling_index(unsigned int k, unsigned int m) {
	if (k == ROOT_NODE) {
		return 0;
	}
	return (k-1)%m;
}

/**
 * Function that returns the filename which contains node i.
 */
char *get_file_from_node(unsigned int k) {
	char *str = xmalloc(sizeof(char)*17);
	if (snprintf(str, 17, "hstore%05u.heap", ((k - (k % F))/F)) <= 0) {
		error("Unable to format filename from node.", __LINE__, __FILE__);
	}
	return str;
}

/**
 * Get the offset into the current file.
 */
uint64_t get_file_offset_from_node(external_heap_t *h, unsigned int k) {
	return (k % h->F) * h->node_size;
}

/**
 * Get number of elements in node.
 */
uint32_t get_node_length(external_heap_t *h, unsigned int k) {
	if (k == ROOT_NODE) {
		assert(h->root_length <= h->m*h->P);
		return h->root_length;
	} else if (k == h->last_leaf) {
		assert(h->last_leaf_length <= h->m*h->P);
		return h->last_leaf_length;
	}

	unsigned int parent = get_parent(k, h->m);
	char *parent_file = get_file_from_node(parent);
	uint64_t parent_offset = get_file_offset_from_node(h, parent) +
		get_sibling_index(k, h->m);

	int fd = h->in->open(parent_file, parent_offset, 1, 1, STREAM_FORWARDS);

	uint32_t length = h->in->next(fd);

	h->in->close(fd);

	free(parent_file);
	assert(length <= h->m*h->P);

	return length;
}

/**
 * Update number of elements in node.
 */
void update_node_length(external_heap_t *h, unsigned int k, uint32_t length) {
	if (k == h->last_leaf) {
		h->last_leaf_length = length;
	}

	if (k == ROOT_NODE) {
		h->root_length = length;
		return;
	}

	unsigned int parent = get_parent(k, h->m);
	char *parent_file = get_file_from_node(parent);
	uint64_t parent_offset = get_file_offset_from_node(h, parent) +
		get_sibling_index(k, h->m);

	int fd = h->out->create(parent_file, parent_offset, 1, 1, STREAM_FORWARDS);

	h->out->write(fd, length);

	h->out->close(fd);

	free(parent_file);
}

/**
 * Update number of elements in a node's children.
 */
void update_child_lengths(external_heap_t *h, unsigned int k,
		uint32_t *lengths, unsigned int child_count) {
	unsigned int i;
	char *node_file = get_file_from_node(k);
	uint64_t offset = get_file_offset_from_node(h, k);
	unsigned int first_child = get_first_child(k, h->m);

	int fd = h->out->create(node_file, offset, child_count, BUFFER, 
			STREAM_FORWARDS);

	for (i = 0; i < child_count; i++) {
		h->out->write(fd, lengths[i]);

		if ((first_child + i) == h->last_leaf) {
			h->last_leaf_length = lengths[i];
			break;
		}
	}

	h->out->close(fd);

	free(node_file);
}

void update_root_buffer(external_heap_t *h) {
	if (h->root_length == 0) {
		return;
	}

	unsigned int length = h->P;

	if (h->root_length < h->P) {
		length = h->root_length;
	}

	char *filename = get_file_from_node(ROOT_NODE);
	uint64_t offset = get_file_offset_from_node(h, ROOT_NODE)
		+ h->node_header_size + h->root_length - length;
	int fd = h->in->open(filename, offset, length, BUFFER, STREAM_FORWARDS);

	unsigned int i = 0;
	while (!h->in->eos(fd)) {
		h->r_buffer[i++] = h->in->next(fd);
	}

	h->in->close(fd);

	free(filename);

	// Update root buffer index
	h->root_buffer_length = length;
}

/**
 * Get the minimum element from the insert buffer
 */
static inline uint32_t get_insert_buffer_min(external_heap_t *h) {
	return internal_heap_front(h->i_buffer);
}

/**
 * Pop the minimum element from the insert buffer
 */
static inline void pop_insert_buffer_min(external_heap_t *h) {
	internal_heap_pop(h->i_buffer);
}

__attribute__ ((visibility("default")))
external_heap_t *heap_init(uint64_t mem_reserved, uint64_t mem_size, 
		in_stream *in, out_stream *out) {
	external_heap_t *h = xmalloc(sizeof(external_heap_t));
	size_t size = sizeof(uint32_t);
	extern int PAGE_SIZE;
	extern int FANOUT;

	h->in = in;
	h->out = out;

	h->M = (mem_size-mem_reserved)/size;

	if (PAGE_SIZE != 0) {
		h->P = PAGE_SIZE/size;
	} else if (FANOUT != 0) {
		h->P = get_pagesize(h->M, FANOUT)/size;
	} else {
		// Default value of page size 4096B = 4KiB.
		h->P = 4096/size;
	}

	// Calculate fanout.
	h->m = get_fanout(h->M, h->P);

	// Calculate nodes per file.
	F = get_nodes_per_file(h->M, h->P, h->m);
	h->F = F;

	// Allocate merge buffer.
	h->m_buffer = xmalloc( ((h->m+1)*h->P) * size );

	// Allocate insert buffer.
	h->i_buffer = xmalloc(sizeof(internal_heap_t));
	internal_heap_init(h->i_buffer, (h->m*h->P), internal_min_heap_comparer); 

	// Allocate page sized buffer for the minimum elements.
	h->r_buffer = xmalloc(h->P * size);

	h->node_header_size = h->m;
	h->node_size = h->m * h->P + h->node_header_size;
	h->last_leaf = 0;
	h->last_leaf_length = 0;
	h->root_length = 0;
	h->root_buffer_length = 0;
	h->count = 0;

	if (BUFFER == 0) {
		BUFFER = h->P;
	}

	return h;
}

__attribute__ ((visibility("default")))
void heap_destroy(external_heap_t *h) {
	if (h->m_buffer != NULL) {
		free(h->m_buffer);
	}

	if (h->i_buffer != NULL) {
		internal_heap_term(h->i_buffer);
		free(h->i_buffer);
	}

	if (h->r_buffer != NULL) {
		free(h->r_buffer);
	}

	free(h);

	return;
}

void bubble_up(external_heap_t *h, unsigned int node) {
	if (node == ROOT_NODE) {
		update_root_buffer(h);
		return;
	}

	// Bubble up from L to F;
	char *node_file = get_file_from_node(node);
	uint64_t node_offset = get_file_offset_from_node(h, node) 
		+ h->node_header_size;

	unsigned int parent = get_parent(node, h->m);
	char *parent_file = get_file_from_node(parent);
	uint64_t parent_offset = get_file_offset_from_node(h, parent)
		+ h->node_header_size;

	// Find end of parent and node
	uint32_t node_length = get_node_length(h, node);
	uint32_t parent_length = get_node_length(h, parent);

	// Load leftmost page of F ...
	unsigned int i = 0, j = 0;
	int parent_in_fd = h->in->open(parent_file, parent_offset, 
			parent_length, BUFFER, STREAM_FORWARDS);

	while (!h->in->eos(parent_in_fd) && i < h->P) {
		h->m_buffer[i] = h->in->next(parent_in_fd);
		i++;
	}

	assert(i <= h->P);

	int node_fd = h->in->open(node_file, node_offset, 
			node_length, BUFFER, STREAM_FORWARDS);

	assert(node_length <= h->m*h->P);
	assert(node_length > 0);
	assert(node_length + i <= (h->m+1)*h->P);

	// ... and entire L into merge buffer
	i = ((h->m+1)*h->P)-node_length;
	while (!h->in->eos(node_fd)) {
		h->m_buffer[i] = h->in->next(node_fd);
		i++;
	}

	unsigned int node_mem_end = i;

	h->in->close(node_fd);

	unsigned int parent_mem_start = 0;
	unsigned int parent_index = parent_mem_start;
	uint32_t parent_prev = h->m_buffer[parent_index++];
	int parent_read_index = parent_length;

	unsigned int node_start = ((h->m+1)*h->P)-node_length;
	unsigned int node_index;
	uint32_t node_prev;

	// Binary Search to find how many elements we at least need to put in L.
	// When done, l + 1 - node_start is the amount of elements we can't copy
	// to F.
	int64_t l = node_start - 1, r = node_start + node_length;
	while (r-l > 1) {
		int64_t mid = l + (r-l)/2;
		if (h->m_buffer[mid] > parent_prev) {
			l = mid;
		} else {
			r = mid;
		}
	}

	node_index = l + 1;

	// Check if all elements have to remain in L, if they do, just return.
	if (node_index >= node_start + node_length) {
		h->in->close(parent_in_fd);
		free(node_file);
		free(parent_file);
		return;
	}

	// Save the first element in m_buffer we have to merge.
	node_prev = h->m_buffer[node_index++];

	// h := #elements in F + L that have higher priority than the lowest in F.
	// Node index is offset by h->P, remember.
	unsigned int elements_left = node_length - ((node_index-1) - node_start)
		+ parent_length;

	unsigned int elements_in_L = (node_index-1) - node_start;

	unsigned int merge_to_L = 0;

	// Put (r - min(h, m*P)) elements in L.
	if (elements_left > h->m*h->P || (node != h->last_leaf && elements_in_L <
				h->m*h->P/2)) {

		if (elements_left > h->m * h->P) {
			merge_to_L = elements_left - h->m*h->P;
		}

		if (merge_to_L + elements_in_L < (h->m*h->P)/2) {
			merge_to_L = (h->m * h->P)/2 - elements_in_L;
		}

		// Open the L output file (the child file).
		int node_out_fd = h->out->create(node_file, 
				(node_offset+((node_index-1)-node_start)), merge_to_L, BUFFER, 
				STREAM_FORWARDS);

		for (i = 0; i < merge_to_L; i++) {
			if ( (node_index-1-node_start) >= node_length || 
					(parent_prev > node_prev && parent_read_index > 0)) {
				// Write from F to L
				h->out->write(node_out_fd, parent_prev);

				// If F page is done, read new
				if (parent_index >= h->P) {
					parent_index = parent_mem_start;

					j = parent_mem_start;
					while (!h->in->eos(parent_in_fd) && j < h->P) {
						h->m_buffer[j++] = h->in->next(parent_in_fd);
					}
				}

				parent_read_index--;
				parent_prev = h->m_buffer[parent_index++];
			} else {
				// Write from L to L
				h->out->write(node_out_fd, node_prev);
				if ( (node_index-1-node_start) < node_length ) {
					node_prev = h->m_buffer[node_index++];
				}
			}
		}

		h->out->close(node_out_fd);
	}

	// Create output to F
	int parent_out_fd = h->out->create(parent_file, parent_offset, 
			(elements_left - merge_to_L), BUFFER, STREAM_FORWARDS); 

	// Load rest of F into memory
	j = h->P;

	while (!h->in->eos(parent_in_fd)) {
		h->m_buffer[j++] = h->in->next(parent_in_fd);
	}

	// Merge rest into F
	
	for (i = 0; i < elements_left - merge_to_L; i++) {
		// We check that node_index is greater than node_mem_end because 
		// node_index is counted one extra up at prev_node.
		if (node_index-1 >= node_mem_end || 
				(parent_prev >= node_prev && parent_read_index > 0)) {
			// Write from F to F
			h->out->write(parent_out_fd, parent_prev);
			parent_read_index--;
			parent_prev = h->m_buffer[parent_index++];
		} else {
			// Write from L to F
			h->out->write(parent_out_fd, node_prev);
			if (node_index < node_mem_end) {	
				node_prev = h->m_buffer[node_index];
			}
			node_index++;
		}
	}

	// Close files
	h->out->close(parent_out_fd);
	h->in->close(parent_in_fd);

	free(parent_file);
	free(node_file);

	// Update endpoints
	update_node_length(h, node, node_length + parent_length - 
			(elements_left - merge_to_L));
	update_node_length(h, parent, elements_left - merge_to_L);

	// Bubble Up
	// TODO 1: Blubble up in a non-recursive manner.
	// TODO 2: Only bubble up if we need to. (Keep tally of the smallest in F).
	//         If F do not get updated we can stop the recursion.
	bubble_up(h, parent);
}

// Insert element v into the heap.
__attribute__ ((visibility("default")))
void heap_insert(external_heap_t *h, uint32_t v) {
	unsigned int i, c;

	h->count++;

	internal_heap_push(h->i_buffer, v);

	if (h->i_buffer->count < h->P*h->m) {
		return;
	}

	/*
	 * Flush stuff to disk.
	 */

	// Sort i_buffer.
	heapsort((Item *)h->i_buffer->data, 0, h->i_buffer->count - 1);

	int fd;
	char *filename;
	uint64_t offset;

	// Elements we want to write to last leaf
	uint64_t count = h->m*h->P - h->last_leaf_length;

	// Empty as much of the i_buffer as possible into the last leaf.
	if (count > 0) {
		// Find a file to write to.
		filename = get_file_from_node(h->last_leaf);

		// Offset to last leaf
		offset = get_file_offset_from_node(h, h->last_leaf)
			+ h->node_header_size;

		// Read all elements from last leaf into merge buffer
		int fd = h->in->open(filename, offset, h->last_leaf_length, BUFFER,
				STREAM_FORWARDS);

		c = 0;
		while (!h->in->eos(fd)) {
			h->m_buffer[c++] = h->in->next(fd);
		}

		h->in->close(fd);

		// Merge existing with insert buffer
		fd = h->out->create(filename, offset, 
				(count + h->last_leaf_length), BUFFER, STREAM_FORWARDS);

		unsigned int ic = count-1;
		unsigned int mc = 0;
		uint32_t ip = h->i_buffer->data[ic]; 
		uint32_t mp; 
		
		if (c == 0) {
			mp = 0;
		} else {
			mp = h->m_buffer[mc++];
		}

		for (i = 0; i < count + h->last_leaf_length; i++) {
			if ( (ip > mp && ic > 0) || c == 0 ) {
				h->out->write(fd, ip);

				if (ic > 0) {
					ip = h->i_buffer->data[ic-1];
					ic--;
				}
			} else {
				h->out->write(fd, mp);
				c--;

				if (c > 0) {
					mp = h->m_buffer[mc];
					mc++;
				}
			}
		}

		h->out->close(fd);

		// Update length of last leaf
		update_node_length(h, h->last_leaf, h->last_leaf_length + count);

		// 4. Bubble up last leaf.
		bubble_up(h, h->last_leaf);

		free(filename);

		// If the entire buffer is inserted in the last leaf
		if (count == h->m*h->P) {
			h->i_buffer->count = 0;

			return;
		}
	}

	// Update pointers to last leaf and update length of last leaf.
	h->last_leaf++;
	update_node_length(h, h->last_leaf, h->m*h->P - count);

	// If i_buffer not empty, create new last leaf, insert remaining
	filename = get_file_from_node(h->last_leaf);

	offset = get_file_offset_from_node(h, h->last_leaf)
		+ h->node_header_size;

	fd = h->out->create(filename, offset, (h->m*h->P-count), BUFFER, 
			STREAM_FORWARDS);

	// Insert elements to new last leaf
	for (i = h->m*h->P; i >= count +1; i--) {
		h->out->write(fd, h->i_buffer->data[i -1]);
	}

	h->out->close(fd);

	// 6. Bubble up this last leaf.
	bubble_up(h, h->last_leaf);

	free(filename);

	h->i_buffer->count = 0;
}

void refill(external_heap_t *h, unsigned int node) {
	char *filename;
	unsigned int offset;
	int fd, lfd;
	unsigned int i, j;
	unsigned int number_of_children = get_number_of_children(h, node);
	unsigned int first_child = get_first_child(node, h->m);

	if (number_of_children > 1) {
		unsigned int child;
		int ifds[number_of_children];
		uint32_t lengths[number_of_children];
		unsigned int length = get_node_length(h, node);

		// Find lenghts of children
		filename = get_file_from_node(node);
		offset = get_file_offset_from_node(h, node);

		fd = h->in->open(filename, offset, number_of_children, BUFFER,
				STREAM_FORWARDS);
		i = 0;
		while (!h->in->eos(fd)) {
			lengths[i++] = h->in->next(fd);
		}
		h->in->close(fd);

		// Move the elements in node to make space for new ones
		lfd = h->out->create(
				filename, 
				offset+h->node_header_size+(h->P*h->m/2), // Offset to new storage
				length,  								  // Elements to move 
				BUFFER, 
				STREAM_FORWARDS
		);
		
		fd = h->in->open(filename, offset+h->node_header_size, 
				length, BUFFER, STREAM_FORWARDS);
		free(filename);

		while (!h->in->eos(fd)) {
			h->out->write(lfd, h->in->next(fd));
		}

		h->in->close(fd);
		h->out->close(lfd);

		// Create input streams for children
		for (i = 0; i < number_of_children; i++) {
			child = i + first_child;
			filename = get_file_from_node(child);
			offset = get_file_offset_from_node(h, child)
				+ h->node_header_size + lengths[i];

			ifds[i] = h->in->open(filename, offset, lengths[i], BUFFER,
					STREAM_BACKWARDS);
			h->in->setbuffer(ifds[i], h->P, h->m_buffer+(i*h->P));
			free(filename);
		}

		filename = get_file_from_node(node);
		offset = get_file_offset_from_node(h, node) + h->node_header_size +
			(h->P*h->m/2);
		int ofd = h->out->create(filename, offset, (h->P*h->m/2), BUFFER, 
				STREAM_BACKWARDS);

		h->out->setbuffer(ofd, h->P, h->m_buffer+(number_of_children*h->P));
		free(filename);

		// Call dmerge
		uint32_t *counts = dmerge(
				number_of_children,			 // Amount of streams to merge
				h->in,						 // Input stream function pointers 
				ifds,						 // List of filedescriptors to read from
				h->out,						 // Output stream function pointers
				ofd,						 // Output stream filedescriptor
				h->P,						 // Buffer size
				((h->P*h->m)/2),			 // Amount of elements to write
				offset-(h->P*h->m/2)+h->P	 // Into output file
		);
		update_node_length(h, node, length+(h->P*h->m/2));

		// Close streams
		for (i = 0; i < number_of_children; i++) {
			h->in->close(ifds[i]);
		}
		h->out->close(ofd);

		// Update lengths
		for (i = 0; i < number_of_children; i++) {
			lengths[i] = lengths[i] - counts[i];
		}

		update_child_lengths(h, node, (uint32_t *)&lengths, number_of_children);

		free(counts);

		for (i = 0; i < number_of_children; i++) {
			if (first_child + i > h->last_leaf) {
				break;
			}

			// -> if children imperfect, refill them
			if (lengths[i] < (h->m*h->P / 2) 
					&& (first_child +i) != h->last_leaf) {
				refill(h, first_child +i);
			} else if (lengths[i] == 0 && (first_child +i) == h->last_leaf
					&& h->last_leaf > ROOT_NODE) {
				unsigned int new_length = get_node_length(h, h->last_leaf-1);
				h->last_leaf--;
				update_node_length(h, h->last_leaf, new_length);
			}
		}
		
		return;
	}

	/////////////////////////////////////////////////
	//               |                             //
	//             .'|'.                           //
	//            /.'|\ \                          //
	// I'm a leaf | /|'.| or I only have one child //
	//             \ |\/                           //
	//              \|/                            //
	//               `                             //
	/////////////////////////////////////////////////

	// If I'm the last leaf, I think it's okay.
	if (node == h->last_leaf) {
		return;
	}

	// if leaf AND imperfect AND not last leaf
	// 	 -> steal from last leaf
	// 	 -> x = leaf length
	// 	 -> s = leaf length + last leaf length
	unsigned int leaf_length = get_node_length(h, node);
	unsigned int s = leaf_length + h->last_leaf_length;
	unsigned int length = 0;

	// Steal from last leaf

	// Calculate the number of elements to steal from last leaf
	if ( s > h->P*h->m ) {
		length = h->P*h->m - leaf_length;
	} else {
		length = h->last_leaf_length;
	}

	// Load last leaf elements into buffer
	filename = get_file_from_node(h->last_leaf);
	offset = get_file_offset_from_node(h, h->last_leaf)
		+ h->node_header_size + (h->last_leaf_length - length);

	fd = h->in->open(filename, offset, length, BUFFER, STREAM_FORWARDS);
	free(filename);

	for (i = 0; i < length; i++) {
		h->m_buffer[i] = h->in->next(fd);
	}

	h->in->close(fd);

	// Load elements from node into buffer
	filename = get_file_from_node(node);
	offset = get_file_offset_from_node(h, node) + h->node_header_size;

	if (leaf_length > 0) {
		lfd = h->in->open(filename, offset, leaf_length, BUFFER, STREAM_FORWARDS);

		for (i = length; i < length+leaf_length; i++) {
			h->m_buffer[i] = h->in->next(lfd);
		}

		h->in->close(lfd);
	} else {
		h->m_buffer[length] = 0;
	}

	// Write to node
	lfd = h->out->create(filename, offset, (length+leaf_length), BUFFER, 
			STREAM_FORWARDS);
	free(filename);

	// Merge last leaf elements with node elements
	i = 0;
	j = 0;

	while ((i + j) < (length + leaf_length)) {
		if ( j >= leaf_length || 
		     (i < length && h->m_buffer[i] >= h->m_buffer[j+length]) ) {
			h->out->write(lfd, h->m_buffer[i]);
			i++;
		} else {
			h->out->write(lfd, h->m_buffer[j+length]);
			j++;
		}
	}

	update_node_length(h, node, length+leaf_length);
	h->out->close(lfd);

	// If last leaf gets empty we have a new last leaf
	if ( s <= h->P*h->m ) {
		unsigned int new_length = get_node_length(h, h->last_leaf-1);
		h->last_leaf--;
		update_node_length(h, h->last_leaf, new_length);
	} else {
		update_node_length(h, h->last_leaf, h->last_leaf_length - length);
	}

	// If last leaf had to few elements, we refill with the new last leaf
	if ( s < h->P*h->m / 2) {
		if ( node != h->last_leaf ) {
			refill(h, node);
		}
	}

	// If the current node and last leaf have same parent we do not have to 
	// bubble up
	if (node == ROOT_NODE || get_parent(node, h->m) == get_parent(h->last_leaf, h->m)) {
		return;
	} 

	// 	   -> if s > mP,
	// 	   	 -> move mP - x
	// 	   -> if mP/2 <= s <= mP
	// 	   	 -> move everything
	// 	   	 -> delete last leaf
	// 	   -> if s < mP/2,
	// 	   	 -> move everything
	// 	   	 -> repeat for new last leaf
	//   -> bubble up
	
	bubble_up(h, node);
}

__attribute__ ((visibility("default")))
uint32_t heap_delete_min(external_heap_t *h) {
	if (h->root_buffer_length == 0 && h->i_buffer->count == 0) {
		error("No elements in heap\n", __LINE__, __FILE__);
	}

	h->count--;

	// Get from root buffer
	uint32_t min_root = 0;

	if (h->i_buffer->count > 0) {
		// Get from insert buffer
		uint32_t min_buffer = get_insert_buffer_min(h);

		if (h->root_buffer_length != 0) {
			min_root = h->r_buffer[h->root_buffer_length -1];

			if (min_root > min_buffer) {
				pop_insert_buffer_min(h);
				return min_buffer;
			}
		} else {
			pop_insert_buffer_min(h);
			return min_buffer;
		}
	} else {
		min_root = h->r_buffer[h->root_buffer_length -1];
	}

	// Root contains the smallest
	update_node_length(h, ROOT_NODE, h->root_length-1);

	h->root_buffer_length--;

	// TODO: If ROOT_NODE is last leaf, then drop this call
	// If root length < mP/2, refill
	if (h->root_length < (h->m*h->P)/2) {
		refill(h, ROOT_NODE);
	}

	// If the returned element is the last in the root buffer
	if (h->root_buffer_length == 0) {
		// Root buffer is empty so we update it.
		update_root_buffer(h);
	}

	return min_root;
}
