#!/usr/bin/env bash
EXE=heap
REPEAT_COUNT=3  # Times we repeat each test
TESTMAX=84    # Maximal number of tests
RESULTS_FOLDER=BENCHMARK_RESULTS
TMP_FOLDER=tmp

DEBUG="1"

# TODO:
# - Check if buffer size is above write/read size?
# - Switch around repetitions, so they are outside.

REPETITION=0
TEST=0

## DEFAULTS START
declare -a Prange=(0)
declare -a mrange=(0)
## DEFAULTS END

PARAM_PRESENT=0

HOST=`hostname | awk -F. '{print $1}'`

# The test being performed
DOING_TEST=""

if [[ ${1} == "P" ]]
then
	# Page sizes
	#                  4KB  16KB  64KB  128KB  512KB  1024KB  4096KB
	declare -a Prange=(4096 16384 65536 131072 524288 1048576 4194304)
	DOING_TEST="Page Size (P)"
elif [[ ${1} == "m" ]]
then
	# Fanout
	declare -a mrange=(50 100 500 1000 5000)
	DOING_TEST="Fanout (m)"
fi

if [[ ${DOING_TEST} == "" ]]
then
	echo "Please specify which test to run."
	echo "	P - Page size test"
	echo "	m - Fanout test"
	exit 0
fi

STAMP="BENCHMARK ${HOST}@`date -u` DOING: ${DOING_TEST}"


# Streams we want to test
declare -a Streams=(3 4);


# Files we try to sort
declare -a Files=("TEST640" "TEST1280" "TEST1920" "TEST2560" "TEST3840" "TEST5120")



function reportProgress() {
        curl monitor.dat7.dk --data "{\"machine\": \"${HOST}\", \"iteration\": ${1},\
                \"iteration_max\": ${2}, \"description\": \"$3\"}" -X POST -H\
                "Content-Type: application/json" --silent -o /dev/null
}

trap "tput cvvis; echo \"\"; reportProgress 1 1 Stopped; exit" INT QUIT

function clearCaches() {
	if [[ ${UID} -eq 0 ]]; then
		echo 3 > /proc/sys/vm/drop_caches
		blockdev --flushbufs /dev/sda
		sync
	else
		echo "NOT ROOOOOOOT!"
	fi
}

function clearTmp() {
	if [[ ${DEBUG} != "1" ]]; then
		rm -f hstore*.heap
	fi
}

function performTest() {
	local S=$1
	local file=$2
	local P=$3
	local m=$4

	# Clear the lines for output
	echo -en "\r"
	tput el

	printf "Performing iteration %d/%d of test %d/%d\n" ${REPETITION} \
		${REPEAT_COUNT} ${TEST} ${TESTMAX}

	reportProgress $(( (${REPETITION} - 1) * ${TESTMAX} + ${TEST} ))\
		$(( ${REPEAT_COUNT} * ${TESTMAX} )) "${file}: S=${S} P=${P} m=${m}"

	# Where we save our data.
	OUTFILE=""

	OUTFILE="${RESULTS_FOLDER}/${S}.${file}.${P}.${m}.csv"

	if [ ! -f ${OUTFILE} ]; then
		echo ${STAMP} >> ${OUTFILE}
	fi

	if [[ ${DEBUG} != "1" ]]; then
		if [[ ${P} == 0 ]]; then
			./memlock/memlock ${EXE} -S ${S} -i ${file} -m ${m} >> ${OUTFILE};
		else
			./memlock/memlock ${EXE} -S ${S} -i ${file} -P ${P} >> ${OUTFILE};
		fi
	else
		echo "${EXE} -S ${S} -i ${file} -P ${P} -m ${m}";
		sleep 0.9
	fi
}

function testWrapper() {
	local S=$1
	local file=$2
	local P=$3
	local m=$4

	clearTmp
	performTest ${S} ${file} ${P} ${m}
	clearCaches
}

TESTMAX=$(( ${#Streams[@]} * ${#Prange[@]} * ${#mrange[@]} * ${#Files[@]} ))
#echo $LEN
#exit

tput civis
mkdir -p ${RESULTS_FOLDER}

for f in "${RESULTS_FOLDER}/*"
do
	echo ${STAMP} | tee -a ${f} >> /dev/null
done

echo ""

REPETITION=0
for r in `seq 1 ${REPEAT_COUNT}`; do
	REPETITION=$(( ${REPETITION} + 1 ))
	TEST=0

	for file in ${Files[@]}; do
		for S in ${Streams[@]}; do
			for P in ${Prange[@]}; do
				for m in ${mrange[@]}; do
					TEST=$(( ${TEST} + 1 ));
					testWrapper ${S} ${file} ${P} ${m}
				done
			done
		done
	done
done

tput cvvis
echo ""
